//=============================================================================
// MYTH_CGC_CardSummonsCore
//=============================================================================

/*:
 * @target MZ
 * @author Isiah Brighton
 * @plugindesc v1.0.4 An extension to Card Game Combat which adds a field cards can be played to.
 * @url https://mythatelier.itch.io/card-summons
 * 
 * @============================================================================
 * Commands
 * ============================================================================
 * 
 * 
 * @command disableCardField
 * @text Disable Card Field
 * @desc Disables the Summons Field and returns to normal CGC battle.
 * 
 * @command enableCardField
 * @text Enable Card Field
 * @desc Enables the Summons Field if it's been disabled.
 * 
 * 
 * ============================================================================
 * Parameters
 * ============================================================================
 * @param gridCoordinates
 * @text Field Grid Settings
 * 
 *     @param gridSettings
 *     @parent gridCoordinates
 *     @type struct<GridSettings>
 *     @text Grid Settings
 *     @default {"gridSize":"{\"width\":\"6\",\"height\":\"2\"}","tileSize":"{\"width\":\"124\",\"height\":\"79\"}","tileOffset":"{\"x\":\"-23\",\"y\":\"0\"}"}
 *     @desc General Settings for the Field Grid Dimensions.
 *     
 *     @param partyFieldSettings
 *     @parent gridCoordinates
 *     @type struct<FieldSettings>
 *     @text Party Field
 *     @desc Settings for the Party Field's appearance and position.
 *     @default {"fieldImage":"field_iso_Party","selectionImage":"field_iso_Select","fieldPosition":"{\"x\":\"298\",\"y\":\"460\"}","firstTileOffset":"{\"x\":\"0\",\"y\":\"0\"}"}
 * 
 *     @param troopFieldSettings
 *     @parent gridCoordinates
 *     @type struct<FieldSettings>
 *     @text Troop Field
 *     @desc Settings for the Troop Field's appearance and position.
 *     @default {"fieldImage":"field_iso_Troop","selectionImage":"field_iso_Select","fieldPosition":"{\"x\":\"420\",\"y\":\"285\"}","firstTileOffset":"{\"x\":\"0\",\"y\":\"0\"}"}
 *     
 *     
 *     
 * 
 * @param fieldZoneInfo
 * @text Field Zone Info
 * @type struct<FieldZone>
 * @desc Extra info about the Field Zone.
 * @default {"cardScale":"0.5","skew":"{\"x\":\"0.0000\",\"y\":\"0.0000\"}","cardRotation":"90","enterSFX":"","exitSFX":"","emptyActions":""}
 *
 * @param Creature Behavior
 * @text Creature Behavior
 * 
 *    @param partyCreatureSettings
 *    @parent Creature Behavior
 *    @text Party Creatures
 *    @type struct<CreatureSettings>
 *    @desc Settings for Creatures on the Party's Field.
 *    @default {"canAttackFirstTurn":"true","tintAfterActions":"true","animationScale":"0.5"}
 *    
 *    @param troopCreatureSettings
 *    @parent Creature Behavior
 *    @text Troop Creatures
 *    @type struct<CreatureSettings>
 *    @desc Settings for Creatures on the Troop's Field.
 *    @default {"canAttackFirstTurn":"true","tintAfterActions":"true","animationScale":"0.5"}
 *    
 * @param Creature Stats
 * @text Creature Stats
 * 
 *    @param creatureStats
 *    @text Stats to Display
 *    @parent Creature Stats
 *    @type struct<StatDisplay>[]
 *    @desc List of Creatures Stats to display on each Creature.
 *    
 *    @param statFontSize
 *    @parent Creature Stats
 *    @text Font Size
 *    @type number
 *    @default 16
 *    @desc Font Size for the value of Creature Stats.
 *    
 *    @param statFontFace
 *    @parent Creature Stats
 *    @text Font Face
 *    @type text
 *    @default GameFont
 *    @desc Name of the Font used to display Stat text for Creatures. Use "GameFont" for default.
 * 
 * 
 * @param Field Skill Window
 * 
 *    @param fieldSkillAnchor
 *    @text Static Position
 *    @parent Field Skill Window
 *    @type struct<Coordinate>
 *    @desc Position of the Field Skill Window, the window that opens when a Creature or Field Card is selected for Skills.
 *    @default {"x":"1184","y":"360"}
 *    
 *    @param fieldSkillRows
 *    @text Visible Rows
 *    @parent Field Skill Window
 *    @type number
 *    @default 4
 *    @min 1
 *    @desc Number of rows that the Field Skill Window can display. Determines its height.
 *    
 *    @param fieldSkillWidth
 *    @text Width
 *    @parent Field Skill Window
 *    @type number
 *    @default 400
 *    @min 1
 *    @desc Width of the Field Skill Window.
 *    
 *    @param followFieldSkill
 *    @text Follow Card?
 *    @parent Field Skill Window
 *    @type boolean
 *    @default true
 *    @desc If ON, Field Skill Window follows the currently selected tile on the Field at an offset.
 * 
 *    @param followFieldSkillOffset
 *    @text Follow Offset
 *    @parent Field Skill Window
 *    @type struct<Coordinate>
 *    @default {"x":"128","y":"-64"}
 *    @desc Offset Value of the Window relative to the selected Tile. Ignore if Follow Conditions is false.
 *    
 * 
 * @param Enemy Status Window
 * @text Enemy Status Window
 * 
 *    @param enemyStatusWindow
 *    @text Position
 *    @parent Enemy Status Window
 *    @type struct<Coordinate>
 *    @desc Position of the Enemy Status Window, the Window that shows the Enemy's health and other stats.
 *    @default {"x":"1280","y":"0"}
 *    
 *    @param enemyStatusSize
 *    @text Size
 *    @parent Enemy Status Window
 *    @type struct<SizeCoordinate>
 *    @desc Width and Height of the Enemy Status Window
 *    @default {"width":"320","height":"180"}
 *    
 *    @param enemyDisplayMode
 *    @text Display Enemy Mode
 *    @parent Enemy Status Window
 *    @type select
 *    @option Mugshot
 *    @option Full Image
 *    @option None
 *    @default Full Image
 *    @desc Sets how the Enemy Status Window visually displays the Enemy. See Plugin Help for more.
 *    
 * @help
 * 
 * 
 * ============================================================================
 * Field Cards
 * ============================================================================
 * 
 * Cards can be placed on the field using the following Card Action:
 *     Move This to Field
 * 
 * Cards have access to three features on the field:
 *     1. Field Skills
 *     2. Start/End of Turn Forced Actions
 *     3. Summoning a Creature
 *     
 * ============================================================================
 * Field Skills
 * ============================================================================
 * 
 * Field Skills are Skills which a Card can execute while on the field.
 * Yes, you are attaching skills to a skill. This means that you can select
 * that card and open a menu of skills to execute. 
 * 
 * Any skill executed this way will use the stats of the owner of that card 
 * for damage formulas. Similarly, any resources those skills cost will be
 * paid by the owner.
 * 
 * Field Skills can be set through the following notetag structure:
 * 
 *    <Field Skills>
 *    Skill X
 *    Skill X Limit Y Per Turn
 *    Skill X Limit Y Per Battle
 *    </Field Skills>
 * 
 * Replace 'X' with a Skill ID.
 * (Optional) replace 'Y' with the amount of times a skill can be used within
 * a turn/battle.
 * 
 * If a card is removed from the field and then placed again, its per-battle
 * skills are refreshed.
 * 
 * 
 * ============================================================================
 * Start/End of Turn Forced Actions
 * ============================================================================
 * 
 * In base CGC, cards in the hand have access to Card Passives which force
 * actions at the start or end of a turn. Here you can do the same thing for
 * the field.
 * 
 * A card can have effects that trigger at the start or end of a turn if they
 * are on the field, using the following notetag structure:
 * 
 *     <Card Passives>
 *     Start of Turn Field: Skill X
 *     End of Turn Field: Skill X
 *     </Card Passives>
 *     
 * Replace 'X' with a Skill ID.
 * 
 * 
 * ============================================================================
 * Creatures
 * ============================================================================
 * 
 * Creatures are battlers that exist on the field. Like actors and enemies,
 * they have stats and skills assigned to them. 
 * 
 * Unlike with Field Skills, any skills a Creature has will use that Creature's
 * stats for damage calculation and cost. However, any Card Actions they 
 * execute or which target them will instead target the battler that they 
 * belong to.
 * 
 * Like how cards are made from the Skills database, Creatures are made from
 * the Enemies database.
 * 
 * Inside a card, you can set it to summon a Creature when it moves to the
 * Field through the following Skill Notetag:
 * 
 *     <Field Creature ID: X>
 *     
 * Replace 'X' with an Enemy ID.
 * 
 * When a card summons a Creature by moving to the field, the card is
 * made invisible, but is still present underneath the Creature. When a
 * Creature's HP reaches 0, its card is moved to the discard. Likewise,
 * if a card is moved off of the field, its Creature dies.
 * 
 * An animation can be played on the Creature when it is summoned by
 * using the following Skill notetag:
 * 
 *     <Summon Animation ID: X>
 *     
 * Replace 'X' with an Animation ID.
 * 
 * The specific placement of a Creature within the Field is relative to
 * what RPG Maker considers its "center point," which is its center
 * bottom in MV and its actual center in MZ. You can offset the location
 * it will spawn in its place in the field using the following notetag:
 * 
 *     <Grid Offset: X Y>
 *     
 * Replace 'X' and 'Y' with numbers, positive or negative.
 * 
 * 
 * Creatures will automatically attack at the end of the turn, using any
 * skills in the Action Patterns for its Enemy data.
 * 
 * Just like Field Cards, Creatures have access to Field Skills using the
 * same notetag structure (inside the Enemies database):
 * 
 *    <Field Skills>
 *    Skill X
 *    Skill X Limit Y Per Turn
 *    Skill X Limit Y Per Battle
 *    </Field Skills>
 *    
 * It should be noted that when the Enemy troop uses Creatures, they
 * don't use these Field Skills. Instead, you should use Action Patterns,
 * or any notetag system other plugins use for Enemy AI.
 *    
 *    
 * ============================================================================
 * Creature-Targeting Forced Actions
 * ============================================================================
 *
 * Creatures have an extra modifier that can be added to (most of) their
 * Forced Actions passives, that makes them the target of the skill.
 *
 *     <Card Passives>
 *     Enter Field: Skill X on This
 *     Start of Turn Field: Skill X on This
 *     End of Turn Field: Skill X on This
 *     </Card Passives>
 *
 * Replace 'X' with a Skill ID.
 * 
 * The actor that summoned them is still the one performing the skill, so
 * ensure that these skills have a scope of 1 Ally.
 * 
 * This allows you to, for instance, grant a buff to a Creature when it enters
 * the Field.
 * 
 * ============================================================================
 * Creature Passives
 * ============================================================================
 * 
 * Creatures have passive notetags which use the following syntax inside the
 * Enemies Database:
 *     <Creature Passives>
 *     passive
 *     passive
 *     </Creature Passives>
 *     
 * The Creature Passives include:
 * 
 *     First Turn Attack: true/false
 * 
 * This will override the default behavior for whether or not Creatures have
 * access to their skills on the same turn they are summoned.
 * Set the default behavior through Plugin Parameters.
 * 
 *     Ignore Action Patterns for Player
 * 
 * This will prevent a Player-side Creature from auto-attacking using
 * Action Patterns. This is useful when you want the same Creature to be
 * used by the party and by the enemy troop - the player's options inside
 * Field Skills and the enemies' options inside Action Patterns.
 * 
 * 
 * 
 * ============================================================================
 * Creatures & Skill Scope
 * ============================================================================
 * 
 * Base RPG Maker allows you to select the Scope of a skill, which determines
 * who can be targeted for it. The addition of Creatures complicates this, and
 * so you should work both with RPG Maker's Scope menu and the system below
 * in tandem to get the scope you want.
 * 
 * By default, all skills can target both Battlers and Creatures, and you can
 * modify this with the following notetags:
 * 
 *     <Card Summons Scope>
 *     Battler
 *     </Card Summons Scope>
 * 
 *     <Card Summons Scope>
 *     Creature
 *     </Card Summons Scope>
 * 
 * In addition, scope can be made to include Battlers or individual
 * Creatures when certain conditions are met, like so:
 * 
 *     <Card Summons Scope>
 *     If [expression]
 *     Creature
 *     </Card Summons Scope>
 *     
 * These if expressions are similar to the If Action in the Core Engine.
 * They evaluate a code expression.
 * 
 * However, in this condition the user has access to the variable "target",
 * which represents any given Creature or Battler, and can be used to evaluate
 * if that specific target is valid for the scope.
 * 
 * For example:
 * 
 *     <Card Summons Scope>
 *     Battler
 *     If target.hp >= target.mhp
 *     Creature
 *     </Card Summons Scope>
 *     
 * In the above example, a skill could target the opponent Battler, and any
 * Creature whose HP is greater than or equal to their Max HP - any Creature
 * who hasn't taken damage.
 * 
 * Creatures that cannot be targeted will be greyed out during selection.
 * 
 * Another example:
 * 
 *     <Card Summons Scope>
 *     If target.friendsUnit().fieldCreatures().length == 0
 *     Battler
 *     If user.mp < user.mmp
 *     Creature
 *     </Card Summons Scope>
 *     
 * The above example could target the opponent Battler only if there are
 * no Creatures on their side of the field. It also can only target
 * Creatures if the user's MP is not full. If neither condition is met
 * there are no valid targets.
 * 
 * 
 * 
 * ============================================================================
 * Card Actions
 * ============================================================================
 * 
 * To select cards on the field, use the following Card Action:
 * 
 *     Select X from Ally Field
 *     Select X from Opponent Field
 *     
 * This works just like the Select X from zoneName Card Action in Action Pack 1,
 * but allows you to specify which field - your own or the opponent's.
 * 
 * You can also use:
 * 
 *    Select X Type typeName from Ally/Opponent Field
 *    
 * If all selected cards have Creatures, you can use the following:
 * 
 *     Use Skill X on Selected
 *     
 * This will make the actor execute the specified skill on each of the
 * Creatures belonging to the selected cards. In other words, you can select
 * a set of creatures, and then use a skill on them.
 * 
 * If a card summons a Creature to the field, you can add this Card Action:
 * 
 *     Use Skill X on This
 * 
 * This will cause the specified skill to execute with the Creature that
 * is being summoned as the target.
 * 
 * 
 *     Battle Log: message
 * 
 * Replace 'message' with any text, and it will send that text to the Battle
 * Log Window to display.
 * 
 * ============================================================================
 * Creatures' Stat Display
 * ============================================================================
 * 
 * Creatures are made up of Enemies, and as such have all of the stat
 * parameters that enemies have. In most games, the relevant stats need to be
 * made visible. 
 * 
 * Using the Stats to Display param, you can specify which stats will appear
 * on the Creature, their coordinates relative to the center of the Enemy
 * Sprite, and whether they display on a Card with a <Field creature ID>
 * notetag.
 * 
 * 
 * You can also specify if the stat will show on all creatures. If you
 * set the Show on All Creatures? to false, the stat will not appear on
 * any Creature or Card by default.
 * 
 * You can override default display behavior by using the following notetags:
 * 
 *    <Show Stat X>
 *    <Hide Stat X>
 * 
 * Replace 'X' with the stat/param associated with the Stat Display.
 * 
 * Stat Displays also have a list of Text Color Conditions, which determine
 * what color the text will be that displays the stat. This allows you to
 * use a system where a number changes color when it lowers.
 * 
 * Each item in the list contains a color and a condition.
 * - The color should be a hex value, though some color names are also 
 *   accepted.
 * - The condition is code which evaluates to true or false.
 * 
 * The Stat Display will use the color of the first condition that returns
 * true in the list.
 * 
 * 
 * ============================================================================
 * Enemy Status Window
 * ============================================================================
 * 
 * The Enemy Status Window is a mirror of the Actor Status Window supplied in
 * MYTH_CGC_PartyUI_TypeA. It inherits a lot of settings from that plugin.
 * However, Enemies do not have Face images like Actors do. For this reason,
 * you have a few options for how to display the appearance of the Enemy within
 * the window:
 * 
 *    - Mugshot
 * 
 * This takes a cropped section of the Enemy sprite and puts it in the corner
 * of the window in the same way the Actor Status Window crops the Actor's
 * Face image.
 * 
 *    - Full Image
 *    
 * This shows the entire Enemy sprite inside the window, behind all text
 * components.
 * 
 *    - None
 * 
 * This does not display the Enemy sprite at all within the Window.
 * 
 * 
 * In the Enemies database, you can adjust the position of the Enemy sprite
 * within the Enemy Status Window using the following notetag:
 * 
 *     <Status Window Offset: X Y>
 * 
 * Replace 'X' and 'Y' with a number, positive or negative.
 * 
 * In Mugshot mode, this will determine what section of the Enemy Sprite is
 * cropped within the corner of the window.
 * In Full Image mode, this will determine the placement of the Enemy Sprite
 * within the window.
 * 
 * Additionally, if you want the Enemy Status Window to show a face just like
 * the Actor Status Window, use the following notetag:
 * 
 *     <Face: fileName index>
 *     
 * Replace 'fileName' with the name of the image in the img/faces folder, and
 * 'index' with the number of the face index you want to use.
 * 
 *
 * ============================================================================
 * Plugin Commands
 * ============================================================================
 *
 * DisableCardField - Disables the Field Zone and all associated Summons 
 *     features.
 * 
 * EnableCardField - Re-enables the Field Zone.
 * 
 * These commands must be called outside of battle.
 * Any card that would move to the Field under normal circumstances instead
 * moves to the Discard while the Field Zone is disabled.
 * 
 * 
 * ============================================================================
 * Debug Mode
 * ============================================================================
 * 
 * Card Summons has a Debug feature which may be useful for lining up 
 * parameters.
 * 
 * If the user hits F11 in battle during a playtest, some debug visuals will
 * appear that show each side's field grid based on the user's parameters, as 
 * well as the "center" of Cards and Creatures.
 * 
 * This can help the user set their parameters to match up with the images
 * they're using to represent the field.
 * 
 * ============================================================================
 * For more information and other features please use the wiki
 * that has been made for this and all related plugins:
 * http://card-game-combat-help.alwaysdata.net/
 * ============================================================================
 * 
 * 
 * ============================================================================
 * Version History
 * ============================================================================
 *
 * v1.0.4 - Fixed Card Summons Scope being unable to reference 'this' in its
 *          conditional eval.
 *          Added utility function card.refundFieldSkilluse(index, amount)
 *          The Require Card Passive now works on Field Creature Skills.
 *          Added Creature compatibility with TurnStartPlus.
 *          Improved timing for End Turn Field Skills.
 *          Added Message Warning infrastructure for multiple languages.
 * 
 * v1.0.3 - Added extra error handling when PartyUI_A is missing/below Card
 *          Summons, and when notetags reference an Enemy ID higher than is
 *          present in the project.
 * v1.0.2 - Fixed bug where Help Window would not open after using a Selection
 *          skill on the Field while using HelpBoxControl.
 * v1.0.1 - Fixed Use Skill on This Card Action targeting the wrong Creature
 *          when the Creature was not the most recently played Card.
 * 
 * v1.0.0 - Finished plugin
 * 
 * ============================================================================
 * Contact Info
 * ============================================================================
 *
 * This tool was developed by folks at MythAtelier LLC. We make Games that Care.
 *
* Need more tools for your project? Be sure to check out our other plugins here:
* https://itch.io/c/1695699/tools-plugins
*
* Have any questions? Run into any bugs? Want to chat? Best place to reach us:
* https://discord.gg/wRk4XHF5tZ
*
* If you like this plugin and want to support us, please give our Patreon a look:
* https://www.patreon.com/mythatelier
*
* Other Places to Find Us:
* - YouTube: https://www.youtube.com/channel/UCQn_1kWmj2iAkNEv9eqL_LA
* - BSky: https://bsky.app/profile/mythatelier.itch.io
* - Steam: https://store.steampowered.com/search/?developer=MythAtelier%2C%20LLC
*
 */

/*~struct~FieldZone:
 *
 * @param cardScale
 * @text Card Scale
 * @type number
 * @decimals 4
 * @desc Scale the Cards will be resized to when moved to this Zone.
 * @default 0.5
 *
 * @param skew
 * @text Card Skew
 * @type struct<Point>
 * @desc Isometric 3D Rotation applied to Cards when moved to this Zone.
 * @default {"x":"0.5000","y":"0.5000"}
 *
 * @param cardRotation
 * @text Card Rotation
 * @type number
 * @min -360
 * @desc 2D Rotation applied to Cards when moved to this Zone (in degrees).
 * @default 0
 *
 * @param enterSFX
 * @text Card Enter SE
 * @type file
 * @dir audio/se/
 * @require 1
 * @desc SE that plays when a Card enters this Zone
 *
 * @param exitSFX
 * @text Card Exit SE
 * @type file
 * @dir audio/se/
 * @require 1
 * @desc SE that plays when a Card exits this Zone
 *
 * @param emptyActions
 * @text Card Actions on Empty
 * @type note
 * @desc List of Card Actions that trigger when the last card is moved from this Zone.
 */

// #region Structs

/*~struct~SizeCoordinate:
 * @param width
 * @text Width
 * @type number
 * @min 1
 * @default 1
 * 
 * @param height
 * @text Height
 * @type number
 * @min 1
 * @default 1
 */

/*~struct~Coordinate:
 * @param x
 * @text X Coordinate
 * @type number
 * @min -10000
 * @default 0
 *
 * @param y
 * @text Y Coordinate
 * @type number
 * @min -10000
 * @default 0
 *
 */

/*~struct~Point:
 * @param x
 * @text X
 * @type number
 * @min -100
 * @default 1
 * @decimals 4
 *
 * @param y
 * @text Y
 * @type number
 * @min -100
 * @default 1
 * @decimals 4
 *
 */

/*~struct~GridSettings:
 * 
 * @param gridSize
 * @text Grid Width/Height
 * @type struct<SizeCoordinate>
 * @desc Amount of tiles wide/high that a Field Grid will be.
 * @default {"width":"6","height":"2"}
 * 
 * @param tileSize
 * @text Tile Width/Height
 * @type struct<SizeCoordinate>
 * @desc Amount of pixels wide/tall that each tile in the Field Grid will be.
 * @default {"width":"124","height":"79"}
 * 
 * @param tileOffset
 * @text Row/Column Offsets
 * @type struct<Coordinate>
 * @desc X/Y Offset of Rows/Columns after the first one on the Field. Recommend using only one of these.
 * @default {"x":"-23","y":"0"}
 */

/*~struct~FieldSettings:
 * 
 * @param fieldImage
 * @text Field Image
 * @desc Image in img/CGC/summons used for the Field
 * 
 * @param selectionImage
 * @text Selection Image
 * @desc Image in img/CGC/summons used to indicate that a tile on a Field is highlighted.
 *  
 * @param fieldPosition
 * @text Field Position
 * @type struct<Coordinate>
 * @desc Coordinates for this Field on screen.
 * 
 * @param firstTileOffset
 * @text First Tile Offset
 * @type struct<Coordinate>
 * @desc Pixel offset between where the Image Coordinates are and where the Tiles are computed.
 * 
 * 
 */

/*~struct~CreatureSettings:
 * 
 * @param canAttackFirstTurn
 * @text Attack First Turn?
 * @type boolean
 * @default true
 * @desc Default Behavior for whether Creatures can act on the same turn they're summoned. Can be overriden by Notetags.
 * 
 * @param tintAfterActions
 * @text Tint When Done?
 * @type boolean
 * @default true
 * @desc Whether Creatures will gain a Dark Tint to indicate they can take no more Actions this Turn.
 * 
 * @param animationScale
 * @text Animation Scale
 * @type number
 * @decimals 2
 * @max 3
 * @min 0.01
 * @default 0.5
 * @desc Scale of Skill Animations when targeting Creatures.
 * 
 */

/*~struct~StatDisplay:
 * 
 * @param param
 * @text Parameter
 * @type string
 * @desc Name of the stat/parameter for this StatDisplay to display. Eg. atk, mhp, hp, etc.
 * 
 * @param coordinate
 * @text Coordinates
 * @type struct<Coordinate>
 * @desc Position relative to the bottom center of the Creature for this Stat to display
 * @default {"x":"0","y":"0"}
 * 
 * @param displayOnCard
 * @text Display on Card?
 * @type boolean
 * @desc If ON, this Stat will show up on Cards with a <Field Creature ID> notetag.
 * @default false
 * 
 *    @param cardCoordinate
 *    @text Card Coordinates
 *    @parent displayOnCard
 *    @type struct<Coordinate>
 *    @desc Position relative to the center of the Card for this Stat to display.
 *    @default {"x":"0","y":"0"}
 * 
 * @param backImage
 * @text Background Image
 * @desc The image to draw behind the text, in img/CGC/summons
 * 
 * @param textConditions
 * @text Text Color Conditions
 * @type struct<TextCondition>[]
 * @default ["{\"textColor\":\"yellow\",\"condition\":\"value < originalValue\"}","{\"textColor\":\"white\",\"condition\":\"true\"}"]
 * @desc Conditions under which the Color of the Text changes.
 * 
 * @param alwaysShow
 * @text Show on All Creatures?
 * @type boolean
 * @default true
 * @desc If OFF, this Stat will only appear for Creatures who use the <Show Stat x> notetag.
 */

/*~struct~TextCondition:
 * @param textColor
 * @text Text Color
 * @type string
 * @desc Hexcode Value for the Text to be under this condition. Some Color names also work.
 * @default white
 * 
 * @param condition
 * @text Condition
 * @type string
 * @default value > 0
 * @desc Eval expression that evaluates to true/false. Use battler.statName, value or originalValue
 */

// #endregion

Input.keyMapper[122] = 'MythDebug';

var Myth = Myth || {};
Myth.CGC = Myth.CGC || {};
Myth.CGC.Summons = Myth.CGC.Summons || {};

//=============================================================================
// Warn Messages
//=============================================================================
// #region Warn Messages
Myth.CGC.Summons.warnMessage = function (index, ...args)
{
	if ($dataSystem == undefined)
	{
		return setTimeout(() =>
		{
			this.warnMessage(index, ...args);
		}, 100);
	}
	const languagePrefix = $dataSystem.locale.slice(0, 2).toLowerCase();
	let text = "";
	let warnMessageMap = {
		ja: this.warnMessageJa,		// Japanese
		zh: this.warnMessageZh,		// Chinese
		ko: this.warnMessageKo,		// Korean
		en: this.warnMessageEn		// English
	};
	if (warnMessageMap[languagePrefix])
		text = warnMessageMap[languagePrefix].call(this, index, ...args);
	else
	{
		text = this.warnMessageEn(index, ...args);	// Default to English for any unsupported language.
	}

	console.warn(text);
	console.warn("MythAtelier Error Code: SUMMON " + index);  // Error code should be the same in every language.
	Myth.Util.openConsole();
	return text;
}

Myth.CGC.Summons.warnMessageEn = function (index, ...args)
{
	let text = "";
	switch (index)
	{
		case 0: // Warning that preempts Error Code CGC 6 or 8 if the Field has been disabled, and therefore the Field zone does not exist.
			let action = args[0];
			text = "Found a Card Action that only works when the Field battle has been abled. Action is:\n"
				+ action + "\n";
			text += "You may get a warning below that an action is not registered. This is because it is not registered outside of the Field battle mode.";
			break;
		case 1: // Eval error
			text = "Error in MYTH_CGC_CardSummonsCore Summons Scope Condition.";
			break;
		case 2:
			{
				let color = args[0];
				let condition = args[1];
				text = "There was an error with the Card Summons Text Condition:\n";
				text += "Color: " + color + "\n";
				text += "Condition: " + condition + "\n";
				text += "The condition could not be evaluated. Check the syntax.";
				break;
			}

	}

	return text;
}

Myth.CGC.Summons.warnMessageJa = function (index, ...args)
{
	return Myth.CGC.Summons.warnMessageEn(index, ...args); // Japanese translation not available yet.
}

Myth.CGC.Summons.warnMessageKo = function (index, ...args)
{
	return Myth.CGC.Summons.warnMessageEn(index, ...args); // Korean translation not available yet.
}

Myth.CGC.Summons.warnMessageZh = function (index, ...args)
{
	return Myth.CGC.Summons.warnMessageEn(index, ...args); // Chinese translation not available yet.
}

// #endregion

Myth.CGC.Summons.debugMode = false;

Myth.Parameters = PluginManager.parameters('MYTH_CGC_CardSummonsCore');

Myth.CGC.Summons.coordinates = {
	gridSettings: JSON.parse(Myth.Parameters.gridSettings),
	troopFieldSettings: JSON.parse(Myth.Parameters.troopFieldSettings),
	partyFieldSettings: JSON.parse(Myth.Parameters.partyFieldSettings),


	fieldSkillWindow: JSON.parse(Myth.Parameters.fieldSkillAnchor),
	fieldSkillRows: Number(Myth.Parameters.fieldSkillRows),
	fieldSkillWidth: Number(Myth.Parameters.fieldSkillWidth),

	followFieldSkill: Myth.Util.tryParse(Myth.Parameters.followFieldSkill, true, "Myth_CGC_CardSummonsCore"),
	followFieldSkillOffset: JSON.parse(Myth.Parameters.followFieldSkillOffset),

	enemyStatusWindow: JSON.parse(Myth.Parameters.enemyStatusWindow),
	enemyStatusSize: JSON.parse(Myth.Parameters.enemyStatusSize),
	enemyDisplayMode: Myth.Parameters.enemyDisplayMode
}
Myth.Util.castMembersToNumber(Myth.CGC.Summons.coordinates);

Myth.CGC.Summons.partyCreatureSettings = JSON.parse(Myth.Parameters.partyCreatureSettings);
Myth.CGC.Summons.troopCreatureSettings = JSON.parse(Myth.Parameters.troopCreatureSettings);
Myth.Util.castMembersToNumber(Myth.CGC.Summons.partyCreatureSettings);
Myth.Util.castMembersToNumber(Myth.CGC.Summons.troopCreatureSettings);

Myth.CGC.zoneInfo.field = JSON.parse(Myth.Parameters.fieldZoneInfo);
Myth.Util.castMembersToNumber(Myth.CGC.zoneInfo.field);

//we don't want to change these
Myth.CGC.zoneInfo.field.name = "field";
Myth.CGC.zoneInfo.field.sheet = "";
Myth.CGC.zoneInfo.field.showCards = "Yes";
Myth.CGC.zoneInfo.field.isSpecial = true;


//helper functions just to make me have to write less stuff
Myth.CGC.Summons.gridWidth = function ()
{
	return this.coordinates.gridSettings.gridSize.width;
}
Myth.CGC.Summons.gridHeight = function ()
{
	return this.coordinates.gridSettings.gridSize.height;
}
Myth.CGC.Summons.tileWidth = function ()
{
	return this.coordinates.gridSettings.tileSize.width;
}
Myth.CGC.Summons.tileHeight = function ()
{
	return this.coordinates.gridSettings.tileSize.height;
}
Myth.CGC.Summons.rowOffset = function ()
{
	return this.coordinates.gridSettings.tileOffset.x;
}
Myth.CGC.Summons.colOffset = function ()
{
	return this.coordinates.gridSettings.tileOffset.y;
}

Myth.CGC.Summons.creatureStats = JSON.parse(Myth.Parameters.creatureStats);
Myth.Util.castMembersToNumber(Myth.CGC.Summons.creatureStats);

Myth.CGC.fontSizes.creatureStat = Number(Myth.Parameters.statFontSize);
Myth.CGC.Summons.creatureStatFont = Myth.Parameters.statFontFace;

//compatibility
if (!Myth.CGC.PartyUI)
{
	setTimeout(() => {
	SceneManager.stop();
	var message = "Myth_CGC_CardSummonsCore requires MYTH_CGC_PartyUI to be loaded before it. \nPlease make sure that MYTH_CGC_PartyUI is above CardSummonsCore in the Plugin Manager.";
	Graphics.printError("Plugin Order Error", message);
	}, 60);
	throw new Error(message);
}

Myth.CGC.PartyUI.doesShowPartyStatus = function ()
{
	if ($gameSystem._cardGridEnabled)
		return false;
	return this.showPartyStatus;
}


Myth.CGC.Summons.getZoneDataByName = Myth.Util.getZoneDataByName;
Myth.Util.getZoneDataByName = function (name)
{
	if (name == 'field')
	{
		if ($gameSystem._cardGridEnabled)
			return Myth.CGC.zoneInfo.field;
		else name == 'discard';
	}
	return Myth.CGC.Summons.getZoneDataByName.call(this, name);
}

Myth.Util.findActiveWindow = function ()
{
	var wins = SceneManager._scene._windowLayer.children;
	for (var i = 0; i < wins.length; i++)
	{
		if (wins[i]._handlers == undefined) continue;
		if (wins[i].isOpenAndActive())
			console.log(wins[i].constructor.name);
	}
}

//=============================================================================
// Plugin Commands
//=============================================================================
// #region Plugin Commands

Myth.CGC.Summons.Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function (command, args)
{
	Myth.CGC.Summons.Game_Interpreter_pluginCommand.call(this, command, args);
	command = command.toLowerCase();
	if (command == 'enablecardfield') $gameSystem.setCardField(true);
	if (command == 'disablecardfield') $gameSystem.setCardField(false);
}

if (Myth.Util.usingMZ)
{

	PluginManager.registerCommand("MYTH_CGC_CardSummonsCore", "enableCardField", args =>
	{
		$gameSystem.setCardField(true);
	});

	PluginManager.registerCommand("MYTH_CGC_CardSummonsCore", "disableCardField", args =>
	{
		$gameSystem.setCardField(false);
	});
}

// #endregion


//=============================================================================
// Game_System
//=============================================================================

Myth.CGC.Summons.Game_System_initCardSettings = Game_System.prototype.initCardSettings;
Game_System.prototype.initCardSettings = function ()
{
	Myth.CGC.Summons.Game_System_initCardSettings.call(this);
	this._cardGridEnabled = true;
};

Game_System.prototype.setCardField = function (enabled)
{
	this._cardGridEnabled = enabled;
};

// For redundancy
Game_System.prototype.setCardGrid = function (enabled)
{
	this.setCardField(enabled);
};


//=============================================================================
// Game_Card
//=============================================================================

Myth.CGC.Summons.Game_Card_initialize = Game_Card.prototype.initialize;
Game_Card.prototype.initialize = function (skillId, origin) {
	Myth.CGC.Summons.Game_Card_initialize.call(this, skillId, origin);
	this.initializeFieldSkills();
};

Game_Card.prototype.initializeFieldSkills = function () {
	this._fieldSkills = [];
	var dataSkill = $dataSkills[this.id()];
	if (!dataSkill) return;

	var dataCreature;
	var hasFieldSkills = false;
	if (dataSkill._fieldSkills)
		hasFieldSkills = true;
	if (dataSkill._fieldCreature) {
		dataCreature = $dataEnemies[dataSkill._fieldCreature];
		if (dataCreature && dataCreature._fieldSkills)
			hasFieldSkills = true;
	}
	if (!hasFieldSkills)
		return;

	if (dataSkill._fieldSkills) {
		for (var i = 0; i < dataSkill._fieldSkills.length; i++) {
			var fieldSkill = dataSkill._fieldSkills[i];
			var index = this._fieldSkills.length;
			this._fieldSkills[index] = Object.assign({}, fieldSkill);
			this._fieldSkills[index].uses = 0;
		}
	}
	if (dataCreature && dataCreature._fieldSkills) {
		for (var i = 0; i < dataCreature._fieldSkills.length; i++) {
			var fieldSkill = dataCreature._fieldSkills[i];
			var index = this._fieldSkills.length;
			this._fieldSkills[index] = Object.assign({}, fieldSkill);
			this._fieldSkills[index].uses = 0;
		}
	}
}

Game_Card.prototype.resetAllFieldSkillUses = function () {
	this.initializeFieldSkills();
	/*	if (!this._fieldSkills) this.initializeFieldSkills();
		for (var i = 0; i < this._fieldSkills.length; i++)
		{
			this._fieldSkills[i].uses = 0;
		}*/
};

Game_Card.prototype.resetTurnFieldSkillUses = function () {
	if (!this._fieldSkills) this.initializeFieldSkills();
	for (var i = 0; i < this._fieldSkills.length; i++) {
		if (this._fieldSkills[i].limitType == 'turn')
			this._fieldSkills[i].uses = 0;
	}
};

Game_Card.prototype.fieldSkill = function (index) {
	if (!this._fieldSkills) this.initializeFieldSkills();
	return this._fieldSkills[index];
};

Game_Card.prototype.spendFieldSkillUse = function (index)
{
	var fieldSkill = this.fieldSkill(index);
	if (fieldSkill)
		fieldSkill.uses++;
}

//Currently a utility function
//TODO:
Game_Card.prototype.refundFieldSkillUse = function (index, amount)
{
	if (amount == undefined) amount = 1;
	var fieldSkill = this.fieldSkill(index);
	if (fieldSkill)
		fieldSkill.uses -= amount;
}

Game_Card.prototype.fieldSkills = function () {
	if (!this._fieldSkills) this.initializeFieldSkills();
	return this._fieldSkills;
};

Game_Card.prototype.selectEffect = function () {
	if (this._fieldCreature)
		this._fieldCreature.select();
	else
		this._selected = true;
}

Game_Card.prototype.deselectEffect = function () {
	if (this._fieldCreature)
		this._fieldCreature.deselect();
	else
		this._selected = false;
};

Game_Card.prototype.reselectEffect = function () {
	if (this.isSelected())
		this._reselected = true;
}

Game_Card.prototype.isSelected = function () {
	return this._selected;
}


//=============================================================================
// Sprite_SkillCard
//=============================================================================
// #region

Myth.CGC.Sprite_SkillCard_initMembers = Sprite_SkillCard.prototype.initMembers;
Sprite_SkillCard.prototype.initMembers = function () {
	Myth.CGC.Sprite_SkillCard_initMembers.call(this);
	this._selectionEffectCount = 0;
	this._statDisplays = [];
	this._debugMode = false;
}

Myth.CGC.Sprite_SkillCard_update = Sprite_SkillCard.prototype.update;
Sprite_SkillCard.prototype.update = function () {
	Myth.CGC.Sprite_SkillCard_update.call(this);
	this.updateSelectionEffect();

	this.updateDebugMode();
};

Sprite_SkillCard.prototype.updateSelectionEffect = function () {
	if (!this._card) return;



	if (this._card.isSelected()) {
		if (this._card._reselected) {
			this._card._reselected = false;
			this._selectionEffectCount = 0;
		}
		else
			this._selectionEffectCount++;
		if (this._selectionEffectCount % 30 < 15) {
			this.setBlendColor([255, 255, 255, 64]);
		} else {
			this.setBlendColor([0, 0, 0, 0]);
		}
	} else if (this._selectionEffectCount > 0) {
		this._selectionEffectCount = 0;
		this.setBlendColor([0, 0, 0, 0]);
	}
};

Myth.CGC.Sprite_SkillCard_setSkill = Sprite_SkillCard.prototype.setSkill;
Sprite_SkillCard.prototype.setSkill = function (skill)
{
	var oldSkill = this._skill;
	Myth.CGC.Sprite_SkillCard_setSkill.call(this, skill);
	if (oldSkill != skill && $gameSystem._cardGridEnabled)
	{
		this.createStatDisplays();
	}
}

Sprite_SkillCard.prototype.createStatDisplays = function ()
{
	for (var i = 0; i < this._statDisplays.length; i++)
	{
		this.removeChild(this._statDisplays[i]);
	}
	this._statDisplays = [];
	var creatureId = this._skill._fieldCreature;
	if (!creatureId) return;

	var tempBattler = new Game_Enemy(creatureId, -1, -1);
	var creatureStats = DataManager.getStatDisplays(creatureId);
	for (var i = 0; i < creatureStats.length; i++)
	{
		if (!creatureStats[i].displayOnCard) continue;
		this._statDisplays[i] = new Sprite_CreatureStat(creatureStats[i], tempBattler);
		this._statDisplays[i].x = creatureStats[i].cardCoordinate.x;
		this._statDisplays[i].y = creatureStats[i].cardCoordinate.y;
		this.addChild(this._statDisplays[i]);
	}
}


Sprite_SkillCard.prototype.updateDebugMode = function ()
{
	if (this._debugMode != Myth.CGC.Summons.debugMode)
	{
		this._debugMode = Myth.CGC.Summons.debugMode;
		if (this._debugDot == undefined)
		{
			var bitmap = new Bitmap(4, 4);
			bitmap.drawCircle(2, 2, 2, 'white');
			this._debugDot = new Sprite(bitmap);
			this._debugDot.anchor.x = 0.5;
			this._debugDot.anchor.y = 0.5;
			this.addChild(this._debugDot);
		}

		this._debugDot.visible = this._debugMode;
		
	}
	if (this._debugDot && this._debugMode)
	{
		this._debugDot.scale.x = 1 / this.scale.x;
		this._debugDot.scale.y = 1 / this.scale.y;
	}
}

Myth.CGC.Summons.Sprite_SkillCard_transformCard = Sprite_SkillCard.prototype.transformCard;
Sprite_SkillCard.prototype.transformCard = function (newCard)
{
	Myth.CGC.Summons.Sprite_SkillCard_transformCard.call(this, newCard);
	if (this._card._fieldCreature)
	{
		var battler = this._card._fieldCreature;
		battler._isTransforming = true;
		battler._newFieldCreature = $dataSkills[newCard.id()]._fieldCreature;
		if (battler._newFieldCreature == undefined)
			battler._newFieldCreature = -1;
		battler._newCard = newCard;
	}
}
// #endregion


//=============================================================================
// Game_Unit
//=============================================================================
// #region

Myth.CGC.Summons.Game_Unit_initialize = Game_Unit.prototype.initialize;
Game_Unit.prototype.initialize = function () {
	Myth.CGC.Summons.Game_Unit_initialize.call(this);
	this._fieldCreatures = [];
}

Myth.CGC.Summons.Game_Unit_onBattleEnd = Game_Unit.prototype.onBattleEnd;
Game_Unit.prototype.onBattleEnd = function () {
	this._fieldCreatures = [];
	Myth.CGC.Summons.Game_Unit_onBattleEnd.call(this);
}

Game_Unit.prototype.fieldCreatures = function () {
	return this._fieldCreatures;
};

Game_Unit.prototype.fieldCards = function () {
	var cards = [];
	this.members().forEach(function (member) {
		cards = cards.concat(member._cardField._data);
	});

	return cards;
};

Game_Unit.prototype.removeFieldCard = function (card) {
	this.members().forEach(function (member) {
		var index = member._cardField.indexOfObject(card);
		if (index > -1) {
			member._cardField.splice(index, 1);
			
			return;
		}
	});
}

Game_Unit.prototype.makeCreatureActions = function () {
	this.fieldCreatures().forEach(function (member) {
		member.makeActions();
	});
};

Game_Unit.prototype.smoothTarget = function (index) {
	if (index < 0) {
		index = 0;
	}
	var member = this.membersAndCreatures()[index];
	return (member && member.isAlive()) ? member : this.aliveMembers()[0];
};

Game_Unit.prototype.smoothDeadTarget = function (index) {
	if (index < 0) {
		index = 0;
	}
	var member = this.membersAndCreatures()[index];
	return (member && member.isDead()) ? member : this.deadMembers()[0];
};

Game_Unit.prototype.membersAndCreatures = function () {
	var totalMembers = this.members().concat(this.fieldCreatures());
	return totalMembers;
}

Game_Unit.prototype.aliveMembersAndCreatures = function () {
	var totalMembers = this.aliveMembers().concat(this.fieldCreatures());
	return totalMembers;
}

Myth.CGC.Summons.Game_Unit_randomTarget = Game_Unit.prototype.randomTarget;
Game_Unit.prototype.randomTarget = function ()
{
	if (BattleManager.fieldTargetMode == 'battler')
		return Myth.CGC.Summons.Game_Unit_randomTarget.call(this);

	var members = [];
	if (BattleManager.fieldTargetMode == 'all')
		members = this.membersAndCreatures();
	else if (BattleManager.fieldTargetMode == 'creature')
		members = this.fieldCreatures();
	else
		members = this.aliveMembers();

	members = this.filterOutUnselectables(null, members);
	var tgrSum = this.tgrSumOfMembers(members);
	var tgrRand = Math.random() * tgrSum;
	var target = null;
	members.forEach(function (member)
	{
		if (member.isAlive())
		{
			tgrRand -= member.tgr;
			if (tgrRand <= 0 && !target)
				target = member;
		}
	});
	return target;
};

Game_Unit.prototype.tgrSumOfMembers = function (members)
{
	return members.reduce((r, member) => r + member.tgr, 0);
}

Game_Unit.prototype.filterOutUnselectables = function (subject, targets)
{
	var remainingTargets = [...targets];
	if (subject == null)
		subject = BattleManager._tempSubject; //bad code do not read.
	if (subject == null)
		subject = BattleManager._subject;
	var user = subject;
	var scope = BattleManager.summonScope;
	if (!scope) return remainingTargets;

	for (var i = 0; i < scope.length; i++)
	{
		if (scope[i].match(/conditional (.+)/i))
		{
			var condition = RegExp.$1;
			var forCreature = scope[i + 1] == "creature";
			for (var j = remainingTargets.length - 1; j >= 0; j--)
			{
				var target = remainingTargets[j];
				if (target.isCreature() && !forCreature)
					continue;
				if (!target.isCreature() && forCreature)
					continue;

				var success = false;
				try
				{
					success = eval(condition);
				}
				catch (error)
				{
					console.warn(error);
					Myth.CGC.Summons.warnMessage(1);
					console.warn(condition);
				}

				if (!success)
					remainingTargets.splice(j, 1);
			}

		}

	}

	return remainingTargets;
}

Game_Unit.prototype.creatureTgrSum = function () {
	return this.fieldCreatures().reduce(function (r, member) {
		return r + member.tgr;
	}, 0);
};

Game_Unit.prototype.allTgrSum = function () {
	return this.membersAndCreatures().reduce(function (r, member) {
		return r + member.tgr;
	}, 0);
}


//Compatibility with YEP_BattleEngineCore
if (Imported.YEP_BattleEngineCore)
{
	Myth.CGC.Summons.Game_Unit_OnTurnStart = Game_Unit.prototype.onTurnStart;
	Game_Unit.prototype.onTurnStart = function ()
	{
		Myth.CGC.Summons.Game_Unit_OnTurnStart.call(this);
		var max = this.fieldCreatures().length;
		for (var i = 0; i < max; ++i)
		{
			var creature = this.fieldCreatures()[i];
			if (creature)
			{
				creature.onTurnStart();
				creature.refresh();
			}
		}
	}

	Myth.CGC.Summons.Game_Unit_refreshMembers = Game_Unit.prototype.refreshMembers;
	Game_Unit.prototype.refreshMembers = function ()
	{
		Myth.CGC.Summons.Game_Unit_refreshMembers.call(this);
		var creatures = this.fieldCreatures();
		for (var i = 0; i < creatures.length; i++)
		{
			var creature = creatures[i];
			if (creature)
				creature.refresh();
		}
	}
}

//Compatibility with YEP_BuffsStatesCore
if (Imported.YEP_BuffsStatesCore)
{
	Myth.CGC.Summons.Game_Unit_processStateEval = Game_Unit.prototype.processStateEval;
	Game_Unit.prototype.processStateEval = function (type)
	{
		Myth.CGC.Summons.Game_Unit_processStateEval.call(this, type);
		var creatures = this.fieldCreatures();
		for (var i = 0; i < creatures.length; i++)
		{
			var creature = creatures[i];
			if (!creature) continue;
			creature.refresh();
			var states = creature.states();
			for (var j = 0; j < states.length; j++)
			{
				var state = states[j];
				if (state)
					creature.customEffectEval(state.id, type);
			}
		}
	}
}

// #endregion

Game_Troop.prototype.leader = function()
{
    return this.members()[0];
}

/*Myth.CGC.Summons.Game_Party_battleMembers = Game_Party.prototype.battleMembers;
Game_Party.prototype.battleMembers = function ()
{
	var members = Myth.CGC.Summons.Game_Party_battleMembers.call(this);
	var creatures = this.fieldCreatures();
	var battleMembers;

	if (BattleManager.fieldTargetMode == 'all')
		battleMembers = members.concat(this.fieldCreatures());
	else if (BattleManager.fieldTargetMode == 'battler')
		battleMembers = members;
	else if (BattleManager.fieldTargetMode == 'creature')
		battleMembers = creatures;

	return battleMembers;
}


Myth.CGC.Summons.Game_Troop_members = Game_Troop.prototype.members;
Game_Troop.prototype.members = function ()
{
	var enemies = Myth.CGC.Summons.Game_Troop_members.call(this);
	var creatures = this.fieldCreatures();
	var members;

	if (BattleManager.fieldTargetMode == 'all')
		members = enemies.concat(this.fieldCreatures());
	else if (BattleManager.fieldTargetMode == 'battler')
		members = enemies;
	else if (BattleManager.fieldTargetMode == 'creature')
		members = creatures;

	return members;
}*/

//=============================================================================
// Game_Battler
//=============================================================================
// #region

Game_Battler.prototype.isCreature = function () {
	return false;
}

Game_Battler.prototype.summonCreature = function (enemyId) {
	//
}

Game_Battler.prototype.isEnabledForTargeting = function ()
{
	return this._enabledForTargeting;
}

Game_Battler.prototype.setEnabledForTargeting = function (enabled, selectHighlightMode)
{
	this._enabledForTargeting = enabled;
}

Myth.CGC.Summons.Game_Battler_onTurnEnd = Game_Battler.prototype.onTurnEnd;
Game_Battler.prototype.onTurnEnd = function () {
	Myth.CGC.Summons.Game_Battler_onTurnEnd.call(this);
	for (var i = 0; i < this._cardField.length; i++) {
		var card = this._cardField.card(i);
		card.resetTurnFieldSkillUses();
	}
}

Myth.CGC.Summons.Game_Battler_getZoneByName = Game_Battler.prototype.getZoneByName;
Game_Battler.prototype.getZoneByName = function (zoneName) {
	if (zoneName.trim().toLowerCase() == "field")
	{
		if ($gameSystem._cardGridEnabled)
			return this._cardField;
		else
			zoneName = "discard";
	}

	return Myth.CGC.Summons.Game_Battler_getZoneByName.call(this, zoneName);
};

Myth.CGC.Summons.Game_Battler_formatCardAction = Game_Battler.prototype.formatCardAction;
Game_Battler.prototype.formatCardAction = function (action)
{
	var originalAction = action;
	action = Myth.CGC.Summons.Game_Battler_formatCardAction.call(this, action);
	if (!$gameSystem._cardGridEnabled && action.match(/field/i))
	{
		action = action.replace(/field/ig, "discard");

		Myth.CGC.Summons.warnMessage(0, originalAction);
		if (action.match(/(party|enemy|troop|ally|opponent) /i))
		{
			action = action.replace(/(party|enemy|troop|ally|opponent) /i, '');
		}
	}

	return action;
}

Myth.CGC.Summons.Game_Battler_performCardAction = Game_Battler.prototype.performCardAction;
Game_Battler.prototype.performCardAction = function (action) {
	var originalAction = action;
	action = this.formatCardAction(action);

	var success = false;
	if (action.match(/battle log:? (.+)/i))
	{
		success = true;
		var message = RegExp.$1;
		var logWindow = SceneManager._scene._logWindow;
		if (logWindow)
		{
			var opponentName = $gameTroop.members()[0].name();
			if (this.isEnemy())
				opponentName = $gameParty.leader().name();
			logWindow.push("addText", message.format(this.name(), opponentName));
		}
	}
	else if ($gameSystem._cardGridEnabled)
	{
		
		success = true;

		if (action.match(/(?:summon creature )(\d+)/i))
		{
			var enemyId = Number(RegExp.$1);
			this.summonCreature(enemyId);
		}
		else if (action.match(/select (\d+)(?: type )(\w+)(?: from )(party|enemy|troop|ally|opponent)(?: field)/i))
		{
			var amount = Number(RegExp.$1);
			var type = RegExp.$2;
			var side = RegExp.$3;
			this.selectFieldCards(amount, type, side);
		}
		else if (action.match(/select (\d+)(?: from )(party|enemy|troop|ally|opponent)(?: field)/i))
		{
			var amount = Number(RegExp.$1);
			var side = RegExp.$2;
			this.selectFieldCards(amount, undefined, side);
		}
		else if (action.match(/select all(?: type )(\w+)(?: from )(party|enemy|troop|ally|opponent)(?: field)/i))
		{
			var type = RegExp.$1;
			var side = RegExp.$2;
			this.selectAllFieldCards(type, side);
		}
		else if (action.match(/select all(?: from )(party|enemy|troop|ally|opponent)(?: field)/i))
		{
			var side = RegExp.$1;
			this.selectAllFieldCards(undefined, side);
		}
		else if (action.match(/use skill (\d+) on selected/i))
		{
			var skillId = Number(RegExp.$1);
			this.forceActionOnSelectedCreatures(skillId);
		}
		else if (action.match(/use skill (\d+) on this/i))
		{
			var skillId = Number(RegExp.$1);
			this.forceActionOnActiveCreature(skillId);
		}
		else success = false;
	}
	else
	{

	}
	if (!success)
		Myth.CGC.Summons.Game_Battler_performCardAction.call(this, originalAction);
};

Game_Battler.prototype.selectFieldCards = function (amount, type, side)
{
	// to be overridden by Game_Enemy and Game_Actor
}

Game_Battler.prototype.selectAllFieldCards = function (type, side)
{
	side = this.getSideFromString(side);
	var unit = side == "party" ? $gameParty : $gameTroop;

	var totalAmount = unit.fieldCards().length;
	if (type)
	{
		totalAmount = unit.aliveMembers()[0].cardsInFieldOfType(type);
	}
	if (totalAmount == 0) return;

	var fieldCards = unit.fieldCards();
	for (var i = 0; i < totalAmount; i++)
	{
		SceneManager._scene._cardSelectionWindow.pushToSelectedCards(fieldCards[i]);
	}

	BattleManager._sideToSelect = side;

	//this.selectFieldCards(fieldSize + 1, type, side); //literally just select over the max
}

Game_Battler.prototype.forceActionOnSelectedCreatures = function (skillId)
{
	var cards = SceneManager._scene._cardSelectionWindow.selectedCards();
	for (var i = 0; i < cards.length; i++)
	{
		var card = cards[i];
		if (!card._fieldCreature) continue;

		var forceAction = {
			actor: this,
			skill: skillId,
			target: card
		}
		BattleManager.__forcedActions.push(forceAction);
	}
}

Game_Battler.prototype.forceActionOnActiveCreature = function (skillId)
{
	var card = BattleManager._currentFieldCard;
	if (card == null)
		card = BattleManager.lastPlayedCard;
	if (!card || !card._fieldCreature) return;
	
	var forceAction = {
		actor: this,
		skill: skillId,
		target: card
	}
	BattleManager.__forcedActions.push(forceAction);
}

Myth.CGC.Summons.Game_Battler_moveCard = Game_Battler.prototype.moveCard;
Game_Battler.prototype.moveCard = function (index, startingZone, endingZone, playSE) {
	if (endingZone.name == "field" && this.atMaxFieldSize()) return;

	if (startingZone.name == "field") {
		var unitZone = this.friendsUnit().fieldCards();
		var referenceCard = unitZone[index];
		var realIndex = startingZone.indexOfObject(referenceCard);
		var owner = this;

		var members = this.friendsUnit().members();
		for (var i = 0; i < members.length; i++) {
			index = members[i]._cardField.indexOfObject(referenceCard);
			if (index != null) {
				owner = members[i];
				break;
			}
		}
		startingZone = owner.getZoneByName(startingZone.name);
		endingZone = owner.getZoneByName(endingZone.name);

		var card = owner.exitCard(startingZone, index, playSE);
		if (card._fieldCreature)
			owner.removeCreature(card);

		if (endingZone) {
			owner.enterCard(endingZone, card, playSE);

			var skillWindow = SceneManager._scene._skillWindow;
			if (skillWindow)
				skillWindow.moveCardSprite(card, index, startingZone, endingZone, owner);
		}


		owner.updateCardVariables();
		return card;
	}

	var card = Myth.CGC.Summons.Game_Battler_moveCard.call(this, index, startingZone, endingZone, playSE);
	if (endingZone.name == "field" && card) {
		card.resetAllFieldSkillUses();
	}
	return card;
};

Game_Battler.prototype.removeCreature = function (card) {
	var index = this.friendsUnit()._fieldCreatures.indexOf(card._fieldCreature);
	this.friendsUnit()._fieldCreatures.splice(index, 1);
	SceneManager._scene._spriteset.removeCreature(card);
	card._fieldCreature = undefined;
}

Game_Battler.prototype.atMaxFieldSize = function ()
{
	var gridWidth = Myth.CGC.Summons.gridWidth();
	var gridHeight = Myth.CGC.Summons.gridHeight();

	var fieldSize = gridWidth * gridHeight;
	var friends = this.friendsUnit().members();
	var totalFieldCards = 0;
	for (var i = 0; i < friends.length; i++)
		totalFieldCards += friends[i]._cardField.length;

	if (totalFieldCards >= fieldSize) 
	{
		return true;
	}

	return false;
};

Myth.CGC.Summons.Game_Battler_removeSelectedCardsSpecial = Game_Battler.prototype.removeSelectedCardsSpecial;
Game_Battler.prototype.removeSelectedCardsSpecial = function () {

	var oldZone = BattleManager._selectedZone;
	if (oldZone.toLowerCase() != "field") {
		return Myth.CGC.Summons.Game_Battler_removeSelectedCardsSpecial.call(this);
	}
	
	var selectionWindow = SceneManager._scene._cardSelectionWindow;
	var fieldWindow = SceneManager._scene._fieldWindow;
	var cards = selectionWindow.selectedCards();
	//var zone = this.getZoneByName(oldZone);

	var unit = BattleManager._sideToSelect == "party" ? $gameParty : $gameTroop;
	var zone = unit.fieldCards();

	for (var i = cards.length - 1; i >= 0; i--) {
		var card = cards[i];
		var index = zone.indexOf(card);
		if (index == -1) continue;

		fieldWindow.giveCardToRemovedSprites(card);

		this.friendsUnit().removeFieldCard(card);
		this.opponentsUnit().removeFieldCard(card);
		this.removedCards++;


	}

	BattleManager._sideToSelect = undefined;
	this.updateCardVariables();
};

Myth.CGC.Summons.Game_Battler_moveSelectedCardsSpecial = Game_Battler.prototype.moveSelectedCardsSpecial;
Game_Battler.prototype.moveSelectedCardsSpecial = function (newZone) {
	var oldZone = BattleManager._selectedZone;
	if (oldZone.toLowerCase() != "field") {
		return Myth.CGC.Summons.Game_Battler_moveSelectedCardsSpecial.call(this, newZone);
	}
	var selectionWindow = SceneManager._scene._cardSelectionWindow;
	var oldZone = BattleManager._selectedZone;

	var startingZone = this.getZoneByName(oldZone);
	var endingZone = this.getZoneByName(newZone);
	var cards = selectionWindow.selectedCards();
	for (var i = 0; i < cards.length; i++) {
		var fieldCards = this.friendsUnit().fieldCards();
		var card = cards[i];
		var index = fieldCards.indexOf(card);
		this.moveCard(index, startingZone, endingZone);
	}
}

Game_Battler.prototype.cardsInFieldOfType = function (type) {
	var amount = 0;
	var cardZone = this.friendsUnit().fieldCards();
	for (var i = 0; i < cardZone.length; i++) {
		var card = cardZone[i];
		var skill = card.id();
		if (DataManager.isSkillOfType(skill, type))
			amount++;
	}

	return amount;
}

Myth.CGC.Summons.Game_Battler_cardsInZoneOfType = Game_Battler.prototype.cardsInZoneOfType;
Game_Battler.prototype.cardsInZoneOfType = function (zone, type) {
	if (zone.toLowerCase() == "field")
		return this.cardsInFieldOfType(type);
	else
		return Myth.CGC.Summons.Game_Battler_cardsInZoneOfType.call(this, zone, type);
}

Game_Battler.prototype.getSideFromString = function (side)
{
	side = side.toLowerCase();
	var newSide = "";
	if (side == "ally")
		newSide = this.isEnemy() ? "troop" : "party";
	if (side == "opponent")
		newSide = this.isEnemy() ? "party" : "troop";

	return newSide;
};

Myth.CGC.Summons.Game_Battler_addCardToZone = Game_Battler.prototype.addCardToZone;
Game_Battler.prototype.addCardToZone = function (skill, zoneName)
{
	if (zoneName.toLowerCase() == "field" && this.atMaxFieldSize())
	{
		return null;
	}
	var card = Myth.CGC.Summons.Game_Battler_addCardToZone.call(this, skill, zoneName);
	if (!$gameSystem._cardGridEnabled)
		return card;

	if (zoneName.toLowerCase() == "field")
	{
		BattleManager.enterZoneForcedActions(this, zoneName, card);

	}
	return card;
};

Myth.CGC.Summons.Game_Battler_forceAction = Game_Battler.prototype.forceAction;
Game_Battler.prototype.forceAction = function (skillId, targetIndex)
{
	var action = new Game_Action(this, true);
	action.setSkill(skillId);
	BattleManager.setFieldTargetMode(action);
	if (action.isForUser())
		targetIndex = this.index();
	Myth.CGC.Summons.Game_Battler_forceAction.call(this, skillId, targetIndex);
}


// #region Useful Getters

Game_Battler.prototype.fieldCount = function (field, type)
{
	if (field == undefined) field = 'ally';
	var side = this.getSideFromString(field);
	var unit = side == "party" ? $gameParty : $gameTroop;
	var cards = unit.fieldCards().filter((card) =>
	{
		if (type == undefined) return true;

		var skill = card.id();
		if (DataManager.isSkillOfType(skill, type))
			return true;
		return false;

	});

	return cards.length;
}

Game_Battler.prototype.creature = function()
{
	var card = this.card();
	if (!card) return null;
	return card._fieldCreature;
}

// #endregion

// #endregion


//=============================================================================
// Game_Enemy
//=============================================================================
// #region

Myth.CGC.Summons.Game_Enemy_initMembers = Game_Enemy.prototype.initMembers;
Game_Enemy.prototype.initMembers = function () {
	Myth.CGC.Summons.Game_Enemy_initMembers.call(this);
	this._cardField = new Game_Cards("field");
	this._cardField.isSpecial = true;
	this._cardSkillsThisTurn = 0;

	this._enabledForTargeting = true;
	this._selectingMode = true;
};

Myth.CGC.Summons.Game_Enemy_addCardToZone = Game_Enemy.prototype.addCardToZone;
Game_Enemy.prototype.addCardToZone = function (skill, zoneName) {
	var card = Myth.CGC.Summons.Game_Enemy_addCardToZone.call(this, skill, zoneName);
	if (!$gameSystem._cardGridEnabled)
		return card;

	if (SceneManager._scene._skillWindow && zoneName.toLowerCase() == "field") {
		SceneManager._scene._skillWindow._getEnemyField = true;
		SceneManager._scene._skillWindow.addCardToZone(card, zoneName, this);
		BattleManager.lastPlayedCard = card;
	}

	this.updateCardVariables();

	return card;
};

Game_Enemy.prototype.canActivateCardSkill = function ()
{
	if (!$gameSystem._cardGridEnabled)
		return false;

	var maxCardActivations = this.enemy()._maxCardSkillsPerTurn;
	if (maxCardActivations == undefined)
		return true;

	return this._cardSkillsThisTurn < maxCardActivations;
}

Game_Enemy.prototype.makeCardPlayAction = function () {
	if (this._actions.length > 0) return false;
	if (this.canMove() && this.canActivateCardSkill()) {
		var fieldCards = this.getFieldCards();
		var actionCards = [];

		// Here we duplicate cards such that there are x copies of a card
		// where x is equal to the amount of usable skills on that card.
		// This way we are able to keep a reference to the chosen card
		// when we choose a random card,
		// while the actual likelihood of any given skill is just as
		// likely as if we had simply picked from a random skill.
		// I did the math. This works.
		for (var i = 0; i < fieldCards.length; i++)
		{
			if (fieldCards[i]._fieldCreature) continue;
			var fieldSkills = fieldCards[i].fieldSkills();
			for (j in fieldSkills) {
				var fs = fieldSkills[j];
				if (fs.uses < fs.limit && this.canUse($dataSkills[fs.skillId]))
					actionCards.push(fieldCards[i]);
			}
		}
		if (actionCards.length > 0) {
			this.selectAllFieldActions(actionCards);
			return true;
		}
	}

	//can no longer activate card skills
	this._cardSkillsThisTurn = 0;
	return false;
};

Game_Enemy.prototype.getFieldCards = function () {
	var cards = [];
	var scene = SceneManager._scene;
	if (!scene instanceof Scene_Battle)
		return cards;

	var spritegroup = scene._spriteset._troopFieldSpritegroup;
	var cardSprites = spritegroup.getBattlerCardSprites(this);
	for (var i = 0; i < cardSprites.length; i++) {
		cards.push(cardSprites[i].card());
	}
	return cards;
};

Game_Enemy.prototype.selectAllFieldActions = function (actionCards) {
	var index = Math.randomInt(actionCards.length);
	var card = actionCards[index];
	var indexList = [];
	var fieldSkills = card.fieldSkills();
	for (i in fieldSkills) {
		var fs = fieldSkills[i];
		if (fs.uses < fs.limit && this.canUse($dataSkills[fs.skillId]))
			indexList.push(i);
	}
	index = Math.randomInt(indexList.length);
	var fieldSkill = fieldSkills[index];
	this._actions.push(new Game_Action(this));
	this.action(this._actions.length - 1).setEnemyAction(fieldSkill);
	BattleManager.setLastFieldSkill(card, index);
	this._cardSkillsThisTurn++;
}

Game_Enemy.prototype.formatCardAction = function (action)
{
	action = Game_Battler.prototype.formatCardAction.call(this, action);
	

	if (action.match(/move this to field/i))
	{
		action = "";
		if (!$gameSystem._cardGridEnabled)
			return action;
		var item = BattleManager._action._item;
		if (item._dataClass == 'skill')
		{
			var skill = $dataSkills[item._itemId];
			action = "add " + item._itemId + " to field";
		}
	}
	return action;
}

Game_Enemy.prototype.selectFieldCards = function (amount, type, side)
{
	Game_Battler.prototype.selectFieldCards.call(this, amount, type, side);
	side = this.getSideFromString(side);
	var unit = side == "party" ? $gameParty : $gameTroop;
	SceneManager._scene.setField(side);

	var totalAmount = unit.fieldCards().length;
	if (type)
	{
		totalAmount = unit.aliveMembers()[0].cardsInFieldOfType(type);
	}
	if (totalAmount == 0) return;
	amount = amount || 1;
	amount = Math.min(amount, totalAmount);

	var fieldCards = unit.fieldCards();
	Myth.Util.shuffleArray(fieldCards);
	for (var i = 0; i < amount; i++)
	{
		SceneManager._scene._cardSelectionWindow.pushToSelectedCards(fieldCards[i]);
	}

	BattleManager._sideToSelect = side;
	BattleManager._selectedZone = "field";

	
}

Myth.CGC.Game_Enemy_refresh = Game_Enemy.prototype.refresh;
Game_Enemy.prototype.refresh = function ()
{
	Myth.CGC.Game_Enemy_refresh.call(this);
	if ($gameSystem._cardGridEnabled)
	{
		var win = SceneManager._scene._enemyStatusWindow;
		if (win)
			win.refresh();
	}

}

Game_Enemy.prototype.setEnabledForTargeting = function (enabled, selectHighlightMode)
{
	this._enabledForTargeting = enabled;
	this._selectHighlightMode = selectHighlightMode;
}

// #endregion


//=============================================================================
// Game_Actor
//=============================================================================
// #region

Myth.CGC.Summons.Game_Actor_initMembers = Game_Actor.prototype.initMembers;
Game_Actor.prototype.initMembers = function () {
	Myth.CGC.Summons.Game_Actor_initMembers.call(this);
	this._cardField = new Game_Cards("field");
	this._cardField.isSpecial = true;

	this._enabledForTargeting = true;
};

Game_Actor.prototype.summonCreature = function (enemyId) {

}

Game_Actor.prototype.friendsCreatures = function () {

}

Game_Actor.prototype.opponentsCreatures = function () {

}

Myth.CGC.Summons.Game_Actor_returnCardsToDeck = Game_Actor.prototype.returnCardsToDeck;
Game_Actor.prototype.returnCardsToDeck = function () {
	this._cardField.clear();
	Myth.CGC.Summons.Game_Actor_returnCardsToDeck.call(this);

	for (var i = 0; i < this._cardDeck.length; i++)
	{
		var card = this._cardDeck.card(i);
		card._fieldCreature = undefined;
	}
}

Game_Actor.prototype.selectFieldCards = function (amount, type, side)
{
	Game_Battler.prototype.selectFieldCards.call(this, amount, type, side);
	BattleManager.requireFieldSelection(this, amount, type, side);
}

Myth.CGC.Summons.Game_Actor_resolveUsedCard = Game_Actor.prototype.resolveUsedCard;
Game_Actor.prototype.resolveUsedCard = function (zoneName)
{
	var card = SceneManager._scene._skillWindow._activeCardSprite.card();
	if (card && zoneName == 'field' && $gameSystem._cardGridEnabled)
	{
		if (!this.atMaxFieldSize())
		{
		var destination = this.getZoneByName(zoneName);
			this.enterCard(destination, card, true);
			
		}
		else
			zoneName = 'discard';
	}
	Myth.CGC.Summons.Game_Actor_resolveUsedCard.call(this, zoneName);
}

Myth.CGC.Summons.Game_Actor_transformSelectedCards = Game_Actor.prototype.transformSelectedCards;
Game_Actor.prototype.transformSelectedCards = function (skillId)
{
	var zoneName = BattleManager._selectedZone;
	if (zoneName == null) return;
	if (zoneName.toLowerCase() != "field")
	{
		Myth.CGC.Summons.Game_Actor_transformSelectedCards.call(this, skillId);
		return;
	}

	var selectionWindow = SceneManager._scene._cardSelectionWindow;
	var cards = selectionWindow.selectedCards();
	var zone = this.getZoneByName(zoneName);
	var spritegroup = SceneManager._scene._skillWindow.getSpritegroupFromName(zoneName);
	var cardSprites = spritegroup.getCardSprites(this);

	for (var i = 0; i < cards.length; i++)
	{
		var card = cards[i];
		var index = zone.indexOfObject(card);
		var newCard = this.transformSpecificCard(card, skillId);
		zone._data[index] = newCard;

		for (var j = 0; j < cardSprites.length; j++)
		{
			if (cardSprites[j].card() == card)
			{
				cardSprites[j].transformCard(newCard);
				break;
			}
		}
	}

}

Myth.CGC.Summons.Game_Actor_checkAllFusionConditions = Game_Actor.prototype.checkAllFusionConditions;
Game_Actor.prototype.checkAllFusionConditions = function ()
{
	var fusionExt = Myth.CGC.Summons.Game_Actor_checkAllFusionConditions.call(this);
	if (fusionExt.id == -1 && $gameSystem._cardGridEnabled)
	{
		fusionExt = this.checkFusionConditions('field');
	}

	return fusionExt;
}

Myth.CGC.Summons.Game_Actor_useIngredients = Game_Actor.prototype.useIngredients;
Game_Actor.prototype.useIngredients = function (ingredients, zone)
{
	for (var i = 0; i < ingredients.length; i++)
	{
		var card = ingredients[i];
		if (card._fieldCreature)
		{
			this.removeCreature(card);
		}
	}
	Myth.CGC.Summons.Game_Actor_useIngredients.call(this, ingredients, zone);
}

// #endregion

//=============================================================================
// Scene_Battle
//=============================================================================
// #region

Myth.CGC.Summons.Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
Scene_Battle.prototype.createAllWindows = function () {
	Myth.CGC.Summons.Scene_Battle_createAllWindows.call(this);

	if (!$gameSystem._cardGridEnabled) return;

	this.createEnemyStatusWindow();

	this.createFieldWindow();
	this.createFieldSkillWindow();

	

	var enemyWin = SceneManager._scene._windowLayer.children.filter(g => g instanceof Window_EnemyStatus)[0];
	var index3 = SceneManager._scene._windowLayer.children.indexOf(enemyWin);
	SceneManager._scene._windowLayer.children.push(SceneManager._scene._windowLayer.children.splice(index3, 1)[0]);

	if (Myth.Util.usingMZ)
	{
		this._enemyWindow.width = 0;
	}
	
}

Scene_Battle.prototype.createEnemyStatusWindow = function ()
{
	this._enemyStatusWindow = new Window_EnemyStatus(this._enemyWindow);
	this._enemyStatusWindow.setHandler('ok', this.onEnemyOk.bind(this));
	this.addWindow(this._enemyStatusWindow);
	this._enemyStatusWindow.refresh();
};

Myth.CGC.Summons.Scene_Battle_createDisplayObjects = Scene_Battle.prototype.createDisplayObjects;
Scene_Battle.prototype.createDisplayObjects = function ()
{
	Myth.CGC.Summons.Scene_Battle_createDisplayObjects.call(this);

	if (!$gameSystem._cardGridEnabled) return;
	BattleManager.setEnemyStatusWindow(this._enemyStatusWindow);
}

Myth.CGC.Summons.Scene_Battle_createEnemyWindow = Scene_Battle.prototype.createEnemyWindow;
Scene_Battle.prototype.createEnemyWindow = function ()
{
	Myth.CGC.Summons.Scene_Battle_createEnemyWindow.call(this);
	if (!$gameSystem._cardGridEnabled) return;
	this._enemyWindow.width = 0;
}

if (Myth.Util.usingMZ)
{
	Myth.CGC.Summons.Scene_Battle_enemyWindowRect = Scene_Battle.prototype.enemyWindowRect;
	Scene_Battle.prototype.enemyWindowRect = function ()
	{
		const rect = Myth.CGC.Summons.Scene_Battle_enemyWindowRect.call(this);
		if ($gameSystem._cardGridEnabled)
			rect.width = 0;
		return rect;
	};
}

//mode can include the following:
// '' - normal browsing on the player's turn
// 'targeting' - selecting a target for a skill
// 'selecting' - selecting a target for a card action (Action Pack 1)
Scene_Battle.prototype.activateFieldWindow = function (side, mode)
{
	if (!$gameSystem._cardGridEnabled) return;

	side = side || "party";
	mode = mode || '';
	this._skillWindow.deactivate();
	//this._skillWindow.deselect();
	//this._skillWindow.deselect();
	this._fieldWindow.setField(side);
	this._fieldWindow.activate();
	if (this._fieldWindow.index() == -1)
		this._fieldWindow.select(0);
	this._fieldWindow.setMode(mode);


/*	this._enemyWindow.deactivate();
	this._enemyWindow.deselect();
	this._actorWindow.deactivate();
	this._actorWindow.deselect();*/


	if (mode == 'selecting')
	{
		// TODO:
		// Figure out why this code A) breaks MV and B) unbreaks MZ
		// if you're a user reading this do NOT call me out on it.
		if (Myth.Util.usingMZ)
		{
			BattleManager._previousPhase = BattleManager._phase;
			BattleManager._phase = 'cardSelection';
		}
	}
	else if (mode == 'targeting')
	{
		//spritegroup.darkenUnselectableSprites();
		//this._spriteset.darkenUnselectableSprites(side);
		var unit = side == "party" ? $gameParty : $gameTroop;
		this.setCreaturesEnabledForTargeting(unit);
	}
	else
	{
		this._lastWindow = 'field';
	}

	this._helpWindow.show();
	//SoundManager.playOk();
};

Scene_Battle.prototype.setField = function(side)
{
	side = side || "party";
	this._fieldWindow.setField(side);
}

Scene_Battle.prototype.setCreaturesEnabledForTargeting = function (unit, enabled)
{
	var allMembers = unit.membersAndCreatures();
	var enabledMembers = unit.filterOutUnselectables(null, unit.membersAndCreatures());

	var mode = enabled ? '' : 'targeting';

	for (var i = 0; i < allMembers.length; i++)
	{
		var isEnabled = enabled;
		if (isEnabled == undefined)
		{
			isEnabled = enabledMembers.includes(allMembers[i]);
		}
		allMembers[i].setEnabledForTargeting(isEnabled, mode);
	}
		// setEnabledForTargeting
}

Scene_Battle.prototype.deactivateFieldWindow = function ()
{
	var mode = this._fieldWindow.mode();
	if (mode == '')
		this._lastWindow = 'skill';
	if (!$gameSystem._cardGridEnabled) return;
	//this._spriteset.undarkenSprites();
	this.setCreaturesEnabledForTargeting($gameTroop, true);
	this.setCreaturesEnabledForTargeting($gameParty, true);

	//just in case
	this._fieldSkillWindow.deactivate();

	this._fieldWindow.deselect();
	this._fieldWindow.setField(null);
	this._fieldWindow.setMode('');
	this._fieldWindow.deactivate();
	//SoundManager.playOk();
};

Scene_Battle.prototype.createFieldWindow = function () {
	this._fieldWindow = new Window_BattleField(this._spriteset);
	this._fieldWindow.setHelpWindow(this._helpWindow);
	this._fieldWindow.setHandler('ok', this.onFieldOk.bind(this));
	this._fieldWindow.setHandler('cancel', this.onFieldCancel.bind(this));
	this.addWindow(this._fieldWindow);

	this._fieldWindow.setStatusWindows(this._actorWindow, this._enemyWindow);
};

Scene_Battle.prototype.createFieldSkillWindow = function () {
	this._fieldSkillWindow = new Window_BattleFieldSkill();
	this._fieldSkillWindow.setHelpWindow(this._helpWindow);
	this._fieldSkillWindow.setHandler('ok', this.onFieldSkillOk.bind(this));
	this._fieldSkillWindow.setHandler('cancel', this.onFieldSkillCancel.bind(this));
	this.addWindow(this._fieldSkillWindow);
};

Scene_Battle.prototype.onFieldOk = function () {

	var card = this._fieldWindow.card();
	var battler = this._fieldWindow.owner();
	var mode = this._fieldWindow.mode();
	if (mode == '')
	{
		BattleManager._currentFieldCard = card;
		var index = this._fieldWindow.index();
		var tileX = this._spriteset._partyGridSprite.getTileX(index) + this._spriteset._partyGridSprite.x;
		var tileY = this._spriteset._partyGridSprite.getTileY(index) + this._spriteset._partyGridSprite.y;
		this._fieldSkillWindow.updatePosition(tileX, tileY);
		this._fieldSkillWindow.open();
		this._fieldSkillWindow.setCard(card, battler);
		this._fieldSkillWindow.activate();
		this._fieldSkillWindow.select(0);
	}
	else if (mode == 'targeting') {
		var action = BattleManager.inputtingAction();
		action.setTarget(card._fieldCreature.index());
		this._fieldWindow.clearPlayerCard();
		this.selectNextCommand();
		//this._fieldWindow.deselect();
		this._fieldSkillWindow.close();
		this._fieldSkillWindow.deactivate();
		this.deactivateFieldWindow();
	}
	else if (mode == 'selecting') {
		var finishedSelecting = this._cardSelectionWindow.pushToSelectedCards(card);
		if (finishedSelecting) {
			this._fieldWindow.endSelectionHighlight();
			this._fieldWindow.deselect();
			this._fieldWindow.setMode('');
			this._cardSelectionWindow.deactivate();
		}
		else {
			this._fieldWindow.activate();
			this._fieldWindow.toggleSelectionHighlight();
		}

	}

};

Scene_Battle.prototype.onFieldCancel = function () {
	var mode = this._fieldWindow.mode();
	if (mode == '') {
		BattleManager._currentFieldCard = null;
		this.deactivateFieldWindow();
	}
	else if (mode == 'targeting') {
		this._enemyWindow.hide();
		if (this._fieldSkillWindow._card)
			this._fieldSkillWindow.activate();

		this._fieldWindow.select(-1);
		this._helpWindow.show();

		this.setCreaturesEnabledForTargeting($gameTroop, true);
		this.setCreaturesEnabledForTargeting($gameParty, true);
	}
	else if (mode == 'selecting') {
		this._fieldWindow.activate();
	}
}

Scene_Battle.prototype.onFieldSkillOk = function () {
	var skill = this._fieldSkillWindow.item();
	var action = BattleManager.inputtingAction();

	var card = this._fieldWindow.getPlayerCard();
	if (card._fieldCreature) {
		action.setSubject(card._fieldCreature);
		card._fieldCreature.makeActions();
		card._fieldCreature._actions[0] = action;
	}
	
	action.setSkill(skill.id);


	BattleManager.actor().setLastBattleSkill(skill);
	BattleManager.setLastFieldSkill(card, this._fieldSkillWindow.index());
	this.onSelectAction();
}

Scene_Battle.prototype.onFieldSkillCancel = function () {
	this._fieldSkillWindow.close();
	this._fieldSkillWindow.setCard(null, null);
	this._fieldSkillWindow.deactivate();
	//this._fieldWindow.activate();
	this.activateFieldWindow();
	var card = this._fieldWindow.getPlayerCard();
	if (card && card._fieldCreature)
	{
		card._fieldCreature.clearActions();
	}
	this._fieldWindow.clearPlayerCard();

	BattleManager.inputtingAction()._fieldSubject = undefined;
};

Myth.CGC.Summons.Scene_Battle_onEnemyCancel = Scene_Battle.prototype.onEnemyCancel;
Scene_Battle.prototype.onEnemyCancel = function ()
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.Scene_Battle_onEnemyCancel.call(this);


	if (this._fieldSkillWindow.isOpen()) {
		this._enemyWindow.hide();
		this._fieldSkillWindow.activate();
		this._helpWindow.show();
		//this.deactivateFieldWindow();
	}
	else {
		this._fieldSkillWindow.deactivate();
		Myth.CGC.Summons.Scene_Battle_onEnemyCancel.call(this);
	}
	BattleManager.setLastFieldSkill(null, null);
};

Myth.CGC.Summons.Scene_Battle_onEnemyOk = Scene_Battle.prototype.onEnemyOk;
Scene_Battle.prototype.onEnemyOk = function () {
	Myth.CGC.Summons.Scene_Battle_onEnemyOk.call(this);

	if (!$gameSystem._cardGridEnabled) return;

	this._fieldSkillWindow.close();
	this._fieldWindow.deselect();
	this._fieldWindow.clearPlayerCard();

	this._enemyWindow.deactivate();
	this._enemyWindow.deselect();

	this._enemyStatusWindow.deselect();
	this._enemyStatusWindow.deactivate();
}

Myth.CGC.Summons.Scene_Battle_isAnyInputWindowActive = Scene_Battle.prototype.isAnyInputWindowActive;
Scene_Battle.prototype.isAnyInputWindowActive = function () {
	var isAnyActive = Myth.CGC.Summons.Scene_Battle_isAnyInputWindowActive.call(this);
	if (!$gameSystem._cardGridEnabled) return isAnyActive;

	return (isAnyActive || this._fieldWindow.active || this._fieldSkillWindow.active);
};

Myth.CGC.Summons.Scene_Battle_onSelectAction = Scene_Battle.prototype.onSelectAction;
Scene_Battle.prototype.onSelectAction = function ()
{
	if ($gameSystem._cardGridEnabled)
	{
		var action = BattleManager.inputtingAction();
		if (!action.needsSelection())
		{
			this._fieldSkillWindow.close();
			this._fieldWindow.deselect();
			this._fieldWindow.clearPlayerCard();
		}
	}
	Myth.CGC.Summons.Scene_Battle_onSelectAction.call(this);
};

// MV uses selectEnemySelection and MZ uses startEnemySelection
Myth.CGC.Summons.Scene_Battle_startEnemySelection = Scene_Battle.prototype.selectEnemySelection;
Scene_Battle.prototype.selectEnemySelection = function ()
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.Scene_Battle_startEnemySelection.call(this);

	var action = BattleManager.inputtingAction();
	BattleManager.setFieldTargetMode(action);
	if (BattleManager.fieldTargetMode == "battler")
		Myth.CGC.Summons.Scene_Battle_startEnemySelection.call(this);
	else
		this.activateFieldWindow("troop", "targeting");

}

if (Myth.Util.usingMZ)
{
	Myth.CGC.Summons.Scene_Battle_startEnemySelection = Scene_Battle.prototype.startEnemySelection;
	Scene_Battle.prototype.startEnemySelection = function ()
	{
		if (!$gameSystem._cardGridEnabled)
			return Myth.CGC.Summons.Scene_Battle_startEnemySelection.call(this);

		var action = BattleManager.inputtingAction();
		BattleManager.setFieldTargetMode(action);
		if (BattleManager.fieldTargetMode == "battler")
			Myth.CGC.Summons.Scene_Battle_startEnemySelection.call(this);
		else
			this.activateFieldWindow("troop", "targeting");
	}
}

Scene_Battle.prototype.startEnemyWindowSelection = function ()
{
	Myth.CGC.Summons.Scene_Battle_startEnemySelection.call(this);

	if (!$gameSystem._cardGridEnabled) return;
	this._enemyStatusWindow.select(0);
	this._enemyStatusWindow.activate();
}

Myth.CGC.Summons.Scene_Battle_selectNextCommand = Scene_Battle.prototype.selectNextCommand;
Scene_Battle.prototype.selectNextCommand = function()
{
	Myth.CGC.Summons.Scene_Battle_selectNextCommand.call(this);
	if (!$gameSystem._cardGridEnabled) return;

	this._fieldSkillWindow.setCard(null, null);
}

Myth.CGC.Summons.Scene_Battle_selectActorSelection = Scene_Battle.prototype.selectActorSelection;
Scene_Battle.prototype.selectActorSelection = function ()
{
	if (!$gameSystem._cardGridEnabled)
	{
		return Myth.CGC.Summons.Scene_Battle_selectActorSelection.call(this);
	}
		

	var action = BattleManager.inputtingAction();
	BattleManager.setFieldTargetMode(action);
	if (BattleManager.fieldTargetMode == "battler")
		Myth.CGC.Summons.Scene_Battle_selectActorSelection.call(this);
	else
		this.activateFieldWindow("party", "targeting");
}

if (Myth.Util.usingMZ)
{
	Myth.CGC.Summons.Scene_Battle_selectActorSelection = Scene_Battle.prototype.startActorSelection;
	Scene_Battle.prototype.startActorSelection = function ()
	{
		if (!$gameSystem._cardGridEnabled)
		{
			return Myth.CGC.Summons.Scene_Battle_selectActorSelection.call(this);
		}

		var action = BattleManager.inputtingAction();
		
		BattleManager.setFieldTargetMode(action);
		if (BattleManager.fieldTargetMode == "battler")
			Myth.CGC.Summons.Scene_Battle_selectActorSelection.call(this);
		else
			this.activateFieldWindow("party", "targeting");
	}
}


Myth.CGC.Summons.Scene_Battle_commandSkill = Scene_Battle.prototype.commandSkill;
Scene_Battle.prototype.commandSkill = function ()
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.Scene_Battle_commandSkill.call(this);

	if (this._lastWindow == 'field')
	{
		//this._skillWindow.refresh();
		this.activateFieldWindow();
		this._skillWindow.deselect();
	}
	else
	{
		//if (this._lastWindow == 'skill')
		return Myth.CGC.Summons.Scene_Battle_commandSkill.call(this);

	}

	
}

// #endregion



//=============================================================================
// BattleManager
//=============================================================================
// #region

// can be:
// battler
// all
// creature
BattleManager.fieldTargetMode = 'battler';
BattleManager.lastFieldSkill = {};

BattleManager.setLastFieldSkill = function (card, index) {
	this.lastFieldSkill.card = card;
	this.lastFieldSkill.index = index;
}

BattleManager.useFieldSkill = function () {
	var card = this.lastFieldSkill.card;
	if (card)
	{
		card.spendFieldSkillUse(this.lastFieldSkill.index);
		//card.fieldSkill(this.lastFieldSkill.index).uses++;
		if (card._fieldCreature) {
			card._fieldCreature.clearActions();
		}
	}
	this.setLastFieldSkill(null, null);
}

BattleManager.playCard = function () 
{
	if (Imported.YEP_BattleEngineCore) 
	{
		this.stopAllSelection();
		this.resetSelection();
	}
	this._subject = this._actionForcedBattler;
	if (!this._subject)
		this._subject = BattleManager.actor();

	
	var action = this._subject.currentAction();
	this._subject = action.subject();
	this._endTheTurn = false;
	if (Imported.YEP_BattleEngineCore && BattleManager.isDTB()) {
		this._ignoreTurnOrderFirstIndex = true;
	}

	this.startAction();
	this.useFieldSkill();
};

Myth.CGC.Summons.BattleManager_donotEndTurn = BattleManager.doNotEndTurn;
BattleManager.doNotEndTurn = function ()
{
	var action = BattleManager.inputtingAction();
	if (action)
		action.clearFieldSubject();
	Myth.CGC.Summons.BattleManager_donotEndTurn.call(this);
	
}

Myth.CGC.Summons.BattleManager_startTurn = BattleManager.startTurn;
BattleManager.startTurn = function ()
{
	if ($gameSystem._cardGridEnabled)
	{
		$gameParty.makeCreatureActions();
		$gameTroop.makeCreatureActions();
	}
	Myth.CGC.Summons.BattleManager_startTurn.call(this);
};

BattleManager.allBattleMembers = function () {
	return $gameParty.membersAndCreatures().concat($gameTroop.membersAndCreatures());
};

Myth.CGC.Summons.BattleManager_getStartCardActions = BattleManager.getStartCardActions;
BattleManager.getStartCardActions = function (actor)
{
	var actions = Myth.CGC.Summons.BattleManager_getStartCardActions.call(this, actor);
	if (!$gameSystem._cardGridEnabled) return actions;
	var creatures = $gameParty.fieldCreatures();

	for (var i = 0; i < creatures.length; i++)
	{
		var newActions = creatures[i].getAllTurnCardActions();
		actions = actions.concat(newActions);
	}

	return actions;
}

Myth.CGC.Summons.BattleManager_makeActionOrders = BattleManager.makeActionOrders;
BattleManager.makeActionOrders = function () {
	var battlers = [];
	if (!this._surprise) {
		var members = [...$gameParty.members()];
		members = this.makeSpeedAndSort(members);
		var creatures = [...$gameParty.fieldCreatures()];
		creatures = this.makeSpeedAndSort(creatures);
		battlers = battlers.concat(members);
		battlers = battlers.concat(creatures);
	}
	if (!this._preemptive) {
		var members = [...$gameTroop.members()];
		members = this.makeSpeedAndSort(members);
		var creatures = [...$gameTroop.fieldCreatures()];
		creatures = this.makeSpeedAndSort(creatures);
		battlers = battlers.concat(members);
		battlers = battlers.concat(creatures);
	}

	this._actionBattlers = battlers;
};

BattleManager.makeSpeedAndSort = function (group) {
	group.forEach(function (battler) {
		battler.makeSpeed();
	});
	group.sort(function (a, b) {
		return b.speed() - a.speed();
	});
	return group;
};

Myth.CGC.Summons.BattleManager_processTurn = BattleManager.processTurn;
BattleManager.processTurn = function ()
{
	if ($gameSystem._cardGridEnabled && !SceneManager._scene._spriteset.isReadyForTurn())
		return;

	var subject = this._subject;
	if (subject.isEnemy() && !subject.__hasCalledForcedActions)
	{
		subject.__hasCalledForcedActions = true;
		this.startTurnFieldForcedActions(subject);
		return;
	}
	Myth.CGC.Summons.BattleManager_processTurn.call(this);
}

Myth.CGC.Summons.BattleManager_endTurn = BattleManager.endTurn;
BattleManager.endTurn = function()
{
	Myth.CGC.Summons.BattleManager_endTurn.call(this);
	if (!$gameSystem._cardGridEnabled) return;

/*	for (var i = 0; i < $gameTroop.members().length; i++)
	{
		var battler = $gameTroop.members()[i];
		battler.__hasCalledForcedActions = false;
		this.endTurnFieldForcedActions(battler);
	}*/

	//TODO: This code was written to fix End of Turn Field Forced Actions
	// for the player and for the enemy, but never worked properly.
	for (var i = 0; i < $gameParty.members().length; i++)
	{
		var battler = $gameParty.members()[i];
		battler.__endTurnForcedActionsCalled = false;
	}
	for (var i = 0; i < $gameTroop.members().length; i++)
	{
		var battler = $gameTroop.members()[i];
		battler.__endTurnForcedActionsCalled = false;
		battler.__hasCalledForcedActions = false;
		//this.endTurnFieldForcedActions(battler);
	}
}

Myth.CGC.Summons.BattleManager_startAction = BattleManager.startAction;
BattleManager.startAction = function ()
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.BattleManager_startAction.call(this);


	this.setFieldTargetMode(this._subject.currentAction());
	Myth.CGC.Summons.BattleManager_startAction.call(this);
	this.setCardTargetsToSummoners();
};

BattleManager.setCardTargetsToSummoners = function () {
	if (!this._cardTargets) return;

	for (var i = 0; i < this._cardTargets.length; i++) {
		if (this._cardTargets[i].isCreature()) {
			this._cardTargets[i] = this._cardTargets[i]._summoner;
		}
	}
}

BattleManager.setFieldTargetMode = function (action) {
	var scope = "all";
	var dataItem = action.item();
	if (dataItem._summonScope)
	{
		var hasCreature = dataItem._summonScope.contains("creature");
		var hasBattler = dataItem._summonScope.contains("battler");
		if (hasCreature && !hasBattler)
			scope = "creature";
		if (hasBattler && !hasCreature)
			scope = "battler";
	}
	this.fieldTargetMode = scope;
	this.summonScope = dataItem._summonScope;
}

Myth.CGC.Summons.BattleManager_endAction = BattleManager.endAction;
BattleManager.endAction = function () {
	this.useFieldSkill();
	Myth.CGC.Summons.BattleManager_endAction.call(this);
	BattleManager._currentFieldCard = null;
	if (this._subject && this._subject.isEnemy()) {
		var success = this._subject.makeCardPlayAction();
		if (success)
		{
			this._actionBattlers.splice(1, 0, this._subject);
		}
	}
};

Myth.CGC.Summons.BattleManager_startTurnForcedActions = BattleManager.startTurnForcedActions;
BattleManager.startTurnForcedActions = function (actor) {
	Myth.CGC.Summons.BattleManager_startTurnForcedActions.call(this, actor);
	if (!$gameSystem._cardGridEnabled) return;

	this.startTurnFieldForcedActions(actor);
}

BattleManager.startTurnFieldForcedActions = function(battler)
{
	for (var i = 0; i < battler._cardField.length; i++) {
		var card = battler._cardField.card(i);
		var skillId = card.id();
		var skill = $dataSkills[skillId];
		if (skill.__forcedActions && skill.__forcedActions.fieldTurnStart) {
			var actions = skill.__forcedActions.fieldTurnStart;
			for (var j = 0; j < actions.length; j++) {
				this.__forcedActions.push({ actor: battler, skill: actions[j] });
			}
		}
	}
}

Myth.CGC.Summons.BattleManager_endTurnForcedActions = BattleManager.endTurnForcedActions;
BattleManager.endTurnForcedActions = function (actor) {
	Myth.CGC.Summons.BattleManager_endTurnForcedActions.call(this, actor);
	if (!$gameSystem._cardGridEnabled) return;

	this.endTurnFieldForcedActions(actor);
}

BattleManager.endTurnFieldForcedActions = function(battler)
{
	for (var i = 0; i < battler._cardField.length; i++) {
		var card = battler._cardField.card(i);
		var skillId = card.id();
		var skill = $dataSkills[skillId];
		if (skill.__forcedActions && skill.__forcedActions.fieldTurnEnd) {
			var actions = skill.__forcedActions.fieldTurnEnd;
			for (var j = 0; j < actions.length; j++) {
				this.__forcedActions.push({ actor: battler, skill: actions[j] });
			}
		}
	}
}

Myth.CGC.Summons.Battlemanager_refreshStatus = BattleManager.refreshStatus;
BattleManager.refreshStatus = function ()
{
	Myth.CGC.Summons.Battlemanager_refreshStatus.call(this);
	if (!$gameSystem._cardGridEnabled) return;
	this._enemyStatusWindow.refresh();
}


BattleManager.setEnemyStatusWindow = function (enemyStatusWindow)
{
	this._enemyStatusWindow = enemyStatusWindow;
}


//Action Pack 1 compatibility
BattleManager.requireFieldSelection = function (actor, amount, type, side) {
	if ($gameTroop.isAllDead())
		return;
	if ($gameParty.isAllDead())
		return;

	if (!$gameSystem._cardGridEnabled)
		return;

	side = actor.getSideFromString(side);
	var unit = side == "party" ? $gameParty : $gameTroop;

	var totalAmount = unit.fieldCards().length;
	if (type) {
		totalAmount = unit.aliveMembers()[0].cardsInFieldOfType(type);
	}
	if (totalAmount == 0) return;


	this._selectedZone = "field";
	this._cardsToSelect = amount || 1;
	this._cardsToSelect = Math.min(this._cardsToSelect, totalAmount);
	this._cardTypeToSelect = type;
	this._sideToSelect = side;


	SceneManager._scene.activateFieldWindow(side, "selecting");
	SceneManager._scene.setCreaturesEnabledForTargeting(unit, true); // <- To update selectability highlights
}

// #endregion

//=============================================================================
// Spriteset_Battle
//=============================================================================
// #region

Myth.CGC.Summons.Spriteset_Battle_createLowerLayer = Spriteset_Battle.prototype.createLowerLayer;
Spriteset_Battle.prototype.createLowerLayer = function () {
	Myth.CGC.Summons.Spriteset_Battle_createLowerLayer.call(this);

	if (!$gameSystem._cardGridEnabled) return;

	this.createFieldGrids();
	this._creatureSprites = [];
}

Spriteset_Battle.prototype.createFieldGrids = function () {
	var troopFieldSettings = Myth.CGC.Summons.coordinates.troopFieldSettings;
	this._troopGridSprite = new Sprite_FieldGrid(troopFieldSettings);
	this._troopGridSprite.x = troopFieldSettings.fieldPosition.x;
	this._troopGridSprite.y = troopFieldSettings.fieldPosition.y;
	this._battleField.addChild(this._troopGridSprite);

	var partyFieldSettings = Myth.CGC.Summons.coordinates.partyFieldSettings;
	this._partyGridSprite = new Sprite_FieldGrid(partyFieldSettings);
	this._partyGridSprite.x = partyFieldSettings.fieldPosition.x;
	this._partyGridSprite.y = partyFieldSettings.fieldPosition.y;
	this._battleField.addChild(this._partyGridSprite);

	this._troopFieldSpritegroup = new Spritegroup_Field();
	this._troopFieldSpritegroup._x = this._troopGridSprite.x;
	this._troopFieldSpritegroup._y = this._troopGridSprite.y;
	this._partyFieldSpritegroup = new Spritegroup_Field();
	this._partyFieldSpritegroup._x = this._partyGridSprite.x;
	this._partyFieldSpritegroup._y = this._partyGridSprite.y;
};

Spriteset_Battle.prototype.createFieldGridBitmap = function (color) {
	var gridWidth = Myth.CGC.Summons.gridWidth();
	var gridHeight = Myth.CGC.Summons.gridHeight();
	var tileWidth = Myth.CGC.Summons.tileWidth();
	var tileHeight = Myth.CGC.Summons.tileHeight();

	var bitmap = new Bitmap(tileWidth * gridWidth + 2, tileHeight * gridHeight + 2);
	for (var i = 0; i <= gridWidth; i++) {
		bitmap.fillRect(i * tileWidth, 0, 2, tileHeight * gridHeight, color);
	}
	for (var i = 0; i <= gridHeight; i++) {
		bitmap.fillRect(0, i * tileHeight, tileWidth * gridWidth, 2, color);
	}

	return bitmap;
}

Myth.CGC.Summons.Spriteset_Battle_update = Spriteset_Battle.prototype.update;
Spriteset_Battle.prototype.update = function () {
	Myth.CGC.Summons.Spriteset_Battle_update.call(this);
	if (!$gameSystem._cardGridEnabled) return;

	this.updateFieldCardPosition(this._troopFieldSpritegroup);
	this.updateFieldCardPosition(this._partyFieldSpritegroup);

	if (Input.isTriggered('MythDebug') && $gameTemp.isPlaytest())
	{
		Myth.CGC.Summons.debugMode = !Myth.CGC.Summons.debugMode;
	}

	//this.sortCreatures();
}

Spriteset_Battle.prototype.isReadyForTurn = function ()
{
	return this.isFinishedMovingCards(this._troopFieldSpritegroup) && this.isFinishedMovingCards(this._partyFieldSpritegroup);
}

Spriteset_Battle.prototype.isFinishedMovingCards = function (spritegroup)
{
	var cardSprites = spritegroup.getCardSprites();
	var cardPositions = spritegroup.cardPositions;
	for (var i = cardSprites.length - 1; i >= 0; i--)
	{
		var cardSprite = cardSprites[i];
		if (!cardSprite) continue;

		var position = cardPositions[i];
		var cardX = this.getFieldCardX(position, spritegroup) + spritegroup._x;
		var cardY = this.getFieldCardY(position, spritegroup) + spritegroup._y;
		if (cardSprite.x != cardX || cardSprite.y != cardY)
		{
			return false;
		}
	}

	return true;
}

Spriteset_Battle.prototype.updateFieldCardPosition = function (spritegroup) {
	var zoneInfo = Myth.CGC.zoneInfo.field;
	var cardSprites = spritegroup.getCardSprites();
	var cardPositions = spritegroup.cardPositions;
	for (var i = cardSprites.length - 1; i >= 0; i--) {
		var cardSprite = cardSprites[i];
		if (!cardSprite) continue;

		var position = cardPositions[i];

		var owner = spritegroup.cardOwner(i);
		var cardX = this.getFieldCardX(position, spritegroup) + spritegroup._x;
		var cardY = this.getFieldCardY(position, spritegroup) + spritegroup._y;
		if (cardSprite.x != cardX || cardSprite.y != cardY) {
			cardSprite.x += (cardX - cardSprite.x) / 10;
			cardSprite.y += (cardY - cardSprite.y) / 10;
			this.finishCardMove(cardSprite, cardX, cardY, owner);
		}

		var targetSkew = zoneInfo.skew;
		if (cardSprite.skew.x != targetSkew.x || cardSprite.skew.y != targetSkew.y) {
			cardSprite.skew.x += (targetSkew.x - cardSprite.skew.x) / 10;
			cardSprite.skew.y += (targetSkew.y - cardSprite.skew.y) / 10;
		}

		var targetAngle = zoneInfo.cardRotation * Math.PI / 180;
		if (cardSprite.rotation != targetAngle)
			cardSprite.rotation += (targetAngle - cardSprite.rotation) / 10;

		var targetScale = zoneInfo.cardScale;
		if (cardSprite.scale.x != targetScale) {
			cardSprite.scale.x += (targetScale - cardSprite.scale.x) / 10;
			cardSprite.scale.y += (targetScale - cardSprite.scale.y) / 10;
		}
	}
};

Spriteset_Battle.prototype.finishCardMove = function (cardSprite, cardX, cardY, owner) {
	var xDist = cardX - cardSprite.x;
	var yDist = cardY - cardSprite.y;
	var totalDistance = Math.sqrt(xDist * xDist + yDist * yDist);
	if (totalDistance < 8)
	{
		this.createCreature(cardSprite, cardX, cardY, owner);
		cardSprite.x = cardX;
		cardSprite.y = cardY;
	}
}

Spriteset_Battle.prototype.createCreature = function (cardSprite, cardX, cardY, owner)
{
	var card = cardSprite.card();
	var dataSkill = $dataSkills[card.id()];
	if (dataSkill._fieldCreature && card._fieldCreature == undefined)
	{
		var creatureY = Math.round(cardY) - this._battleField.y;
		//var creatureY = Math.round(cardY) - this._battleField.y;
		var creatureX = Math.round(cardX) - this._battleField.x;
		card._fieldCreature = new Game_CardCreature(dataSkill._fieldCreature, creatureX, creatureY, card, cardSprite._actor);
		var creatureSprite = new Sprite_CardCreature(card._fieldCreature);
		this._creatureSprites.push(creatureSprite);
		this._battleField.addChild(creatureSprite);
		cardSprite.hide();

		if (owner && owner.isEnemy())
			$gameTroop._fieldCreatures.push(card._fieldCreature);
		else if (owner && owner.isActor())
			$gameParty._fieldCreatures.push(card._fieldCreature);

		card._fieldCreature.makeActions();
		BattleManager._actionBattlers.push(card._fieldCreature);
		
		this.sortCreatures();

		this.requestSummonAnimation(dataSkill, card._fieldCreature);
	}
}

Spriteset_Battle.prototype.sortCreatures = function ()
{
	var creatures = [];
	for (var i = this._battleField.children.length - 1; i >= 0; i--)
	{
		var child = this._battleField.children[i];
		if (child instanceof Sprite_CardCreature)
		{
			creatures.push(child);
			this._battleField.removeChild(child);
		}
	}
	creatures.sort(this._compareCreatureOrder.bind(this));
	for (var i = 0; i < creatures.length; i++)
	{
		this._battleField.addChild(creatures[i]);
	}
};

Spriteset_Battle.prototype._compareCreatureOrder = function (a, b)
{
	if (a.y !== b.y)
	{
		return a.y - b.y;
	} else
	{
		return a._battler._enemyId - b._battler._enemyId;
	}
};

Spriteset_Battle.prototype.requestSummonAnimation = function(dataSkill, target)
{
	if (dataSkill._summonAnimation)
	{
		if (Myth.Util.usingMZ)
		{
			var request = {
				targets: [target],
				animationId: dataSkill._summonAnimation,
				mirror: false
			}
			this.createAnimation(request);
		}
		else
		{
			target.startAnimation(dataSkill._summonAnimation, false, 0);
		}
	}

}

Spriteset_Battle.prototype.getFieldCardX = function (index, spritegroup) {
	var gridSprite = spritegroup == this._partyFieldSpritegroup ? this._partyGridSprite : this._troopGridSprite;
	var tileX = gridSprite.getTileX(index);
	return tileX + (Myth.CGC.Summons.tileWidth() / 2) + this._battleField.x;
}

Spriteset_Battle.prototype.getFieldCardY = function (index, spritegroup) {
	var gridSprite = spritegroup == this._partyFieldSpritegroup ? this._partyGridSprite : this._troopGridSprite;
	var tileY = gridSprite.getTileY(index);

	return tileY + (Myth.CGC.Summons.tileHeight() / 2) + this._battleField.y;
};

Spriteset_Battle.prototype.removeCreature = function (card)
{
	if (!card._fieldCreature) return;
	var creatureSprite = this._creatureSprites.find((sprite) => sprite._battler == card._fieldCreature);
	if (!creatureSprite) return;

	this._battleField.removeChild(creatureSprite);
	var index = this._creatureSprites.indexOf(creatureSprite);
	if (!Myth.Util.usingMZ)
	{
		for (var i = 0; i < creatureSprite._animationSprites.length; i++)
		{
			this._battleField.removeChild(creatureSprite._animationSprites[i]);
		}
	}
	
	this._creatureSprites.splice(index, 1);
}

Spriteset_Battle.prototype.findTargetSprite = function (target)
{
	return this.battlerAndCreatureSprites().find(sprite => sprite.checkBattler(target));
}

Spriteset_Battle.prototype.battlerAndCreatureSprites = function ()
{
	return this.battlerSprites().concat(this._creatureSprites);
}
/*Spriteset_Battle.prototype.updateActors = function ()
{
	var members = $gameParty.battleMembers();
	for (var i = 0; i < this._actorSprites.length; i++)
	{
		if (!members[i] || !members[i].isActor()) return;
		this._actorSprites[i].setBattler(members[i]);
	}
};*/

// #endregion


//=============================================================================
// Sprite_FieldGrid
//=============================================================================
// #region

function Sprite_FieldGrid() {
	this.initialize.apply(this, arguments);
}

Sprite_FieldGrid.prototype = Object.create(Sprite.prototype);
Sprite_FieldGrid.prototype.constructor = Sprite_FieldGrid;

Sprite_FieldGrid.prototype.initialize = function (fieldSettings) {

	this._fieldSettings = fieldSettings;
	var bitmap = ImageManager.loadBitmap("img/CGC/summons/", fieldSettings.fieldImage);
	var tileWidth = Myth.CGC.Summons.tileWidth();
	var tileHeight = Myth.CGC.Summons.tileHeight();
	Sprite.prototype.initialize.call(this, bitmap);

	var selectionBitmap = new Bitmap(tileWidth, tileHeight);
	selectionBitmap.fillRect(2, 2, selectionBitmap.width - 2, selectionBitmap.height - 2, 'white');

	var selectionBitmap = ImageManager.loadBitmap("img/CGC/summons/", fieldSettings.selectionImage);
	this._selectionSprite = new Sprite(selectionBitmap);
	this.addChild(this._selectionSprite);
	this._selectionSprite.visible = false;
	this.__time = 0;
	this._index = 0;
}

Sprite_FieldGrid.prototype.update = function () {
	Sprite.prototype.update.call(this);

	if (SceneManager._scene._fieldWindow && SceneManager._scene._fieldWindow.active)
		this.__time++;
	else
		this.__time = 10;
	this._selectionSprite.opacity = Math.sin(this.__time / 10) * (255 / 2) + (255 / 2);

	if (this._debugGridSprite == undefined)
		this.createDebugGridSprite();
	this._debugGridSprite.visible = Myth.CGC.Summons.debugMode;
}

Sprite_FieldGrid.prototype.createDebugGridSprite = function ()
{
	var gridWidth = Myth.CGC.Summons.gridWidth();
	var gridHeight = Myth.CGC.Summons.gridHeight();
	var tileWidth = Myth.CGC.Summons.tileWidth();
	var tileHeight = Myth.CGC.Summons.tileHeight();
	var rowOffset = Myth.CGC.Summons.rowOffset();
	var colOffset = Myth.CGC.Summons.colOffset();

	var bitmapWidth = gridWidth * tileWidth;
	var offsetWidth = Math.abs(rowOffset * gridHeight);
	bitmapWidth += offsetWidth;
	var bitmapHeight = gridHeight * tileHeight;
	var offsetHeight = Math.abs(colOffset * gridWidth);
	bitmapHeight += offsetHeight;
	var bitmap = new Bitmap(bitmapWidth, bitmapHeight);
	//bitmap.fillAll('black');

	for (var i = 0; i < gridWidth * gridHeight; i++)
	{
		var _x = this.getTileX(i) - this._fieldSettings.firstTileOffset.x;
		if (rowOffset < 0)
			_x += offsetWidth;
		var _y = this.getTileY(i) - this._fieldSettings.firstTileOffset.y;
		if (colOffset < 0)
			_y += offsetHeight;
		var width = tileWidth;
		var height = tileHeight;
		bitmap.fillRect(_x, _y, width, height);
		bitmap.clearRect(_x + 1, _y + 1, width - 2, height - 2);

		bitmap.drawCircle(_x + (tileWidth * 0.5), _y + (tileHeight * 0.5), 4);
	}

	this._debugGridSprite = new Sprite(bitmap);
	this._debugGridSprite.x = this._fieldSettings.firstTileOffset.x;
	if (rowOffset < 0)
		this._debugGridSprite.x -= offsetWidth;
	this._debugGridSprite.y = this._fieldSettings.firstTileOffset.y;
	if (colOffset < 0)
		this._debugGridSprite.y -= offsetHeight;
	this.addChild(this._debugGridSprite);
}

Sprite_FieldGrid.prototype.select = function (index) {
	this._index = index;
	if (this._index > -1) {
		this._selectionSprite.visible = true;
		var x = this.getTileX(index);
		var y = this.getTileY(index);
		this._selectionSprite.x = x
		this._selectionSprite.y = y;	}
	else
		this._selectionSprite.visible = false;
};

Sprite_FieldGrid.prototype.getTileX = function (index) {
	var gridWidth = Myth.CGC.Summons.gridWidth();
	var tileWidth = Myth.CGC.Summons.tileWidth();
	var rowOffset = Myth.CGC.Summons.rowOffset();

	var slotX = index % gridWidth;
	var slotY = Math.floor(index / gridWidth);

	var squareX = (tileWidth * slotX);
	squareX += (slotY * rowOffset);
	
	squareX += this._fieldSettings.firstTileOffset.x;
	return squareX;
}

Sprite_FieldGrid.prototype.getTileY = function (index) {
	var gridWidth = Myth.CGC.Summons.gridWidth();
	var tileHeight = Myth.CGC.Summons.tileHeight();
	var colOffset = Myth.CGC.Summons.colOffset();

	var slotX = index % gridWidth;
	var slotY = Math.floor(index / gridWidth);
	var squareY = (tileHeight * slotY);
	squareY += (slotX * colOffset);
	
	squareY += this._fieldSettings.firstTileOffset.y;
	return squareY;
}

// #endregion


//=============================================================================
// Game_CardCreature
//=============================================================================
// #region

function Game_CardCreature() {
	this.initialize.apply(this, arguments);
}

Game_CardCreature.prototype = Object.create(Game_Enemy.prototype);
Game_CardCreature.prototype.constructor = Game_CardCreature;

Game_CardCreature.prototype.initialize = function (enemyId, x, y, card, summoner) {
	this._card = card;
	this._summoner = summoner;
	Game_Enemy.prototype.initialize.call(this, enemyId, x, y);
	this._firstTurnSpawned = true;
	this._scaleEffect = 0;

	this._selectHighlightMode = '';

};

Game_CardCreature.prototype.summoner = function ()
{
	return this._summoner;
}

Game_CardCreature.prototype.field = function()
{
	return this._summoner._cardField;
}

Game_CardCreature.prototype.friendsUnit = function () {
	if (this._summoner.isEnemy())
		return $gameTroop;
	else
		return $gameParty;
}

Game_CardCreature.prototype.opponentsUnit = function () {
	if (this._summoner.isActor())
		return $gameTroop;
	else
		return $gameParty;
};

Game_CardCreature.prototype.isEnemy = function () {
	return false;
};

Game_CardCreature.prototype.isCreature = function () {
	return true;
};

Game_CardCreature.prototype.index = function () {
	var unit = this.friendsUnit();
	var creatures = unit.fieldCreatures();
	var index = creatures.indexOf(this);
	return unit.members().length + index;
};

Game_CardCreature.prototype.makeActions = function () {
	var creaturePassives = this.enemy()._creaturePassives;
	if (creaturePassives && creaturePassives._ignoreActionPatternsPlayer && this._summoner.isActor())
		Game_Battler.prototype.makeActions.call(this);
	else
		Game_Enemy.prototype.makeActions.call(this);
};

Game_CardCreature.prototype.canAttackFirstTurn = function () {
	var creaturePassives = this.enemy()._creaturePassives;
	if (creaturePassives && creaturePassives._canAttackFirstTurn != undefined)
		return creaturePassives._canAttackFirstTurn;
	else
	{
		if (this.friendsUnit() == $gameParty)
			return Myth.CGC.Summons.partyCreatureSettings.canAttackFirstTurn;

		return Myth.CGC.Summons.troopCreatureSettings.canAttackFirstTurn;
	}
}

Game_CardCreature.prototype.canMove = function ()
{
	if (!this.canMoveThisTurn())
		return false;

	return Game_Enemy.prototype.canMove.call(this);
};

Game_CardCreature.prototype.canMoveThisTurn = function ()
{
	if (this.canAttackFirstTurn() == false)
	{
		if (this._firstTurnSpawned)
			return false;
	}
	return true;
}

Game_CardCreature.prototype.canUse = function (item)
{
	var canUse = Game_Enemy.prototype.canUse.call(this, item);
	if (!canUse) return false;

	return this.cardMeetsUseRequirements(item);
}

Game_CardCreature.prototype.cardMeetsUseRequirements = function (item)
{
	return Game_Actor.prototype.cardMeetsUseRequirements.call(this, item);
}

Game_CardCreature.prototype.onTurnEnd = function () {
	Game_Enemy.prototype.onTurnEnd.call(this);
	this._firstTurnSpawned = false;
};

Game_CardCreature.prototype.performCollapse = function (ignoreMove)
{
	Game_Enemy.prototype.performCollapse.call(this);
	if (!this._summoner) return;
	if (ignoreMove) return;

	//SceneManager._scene._spriteset.removeCardFromField(this._card);
	if (this._summoner.isActor()) {

		var fieldCards = this._summoner.getZoneByName("field");
		var discardCards = this._summoner.getZoneByName("discard");
		var index = fieldCards._data.indexOf(this._card);
		if (index > -1) {
			this._summoner.moveCard(index, fieldCards, discardCards, false);

			var index = $gameParty._fieldCreatures.indexOf(this);
			if (index > -1)
				$gameParty._fieldCreatures.splice(index, 1);
		}
	}
	else if (this._summoner.isEnemy()) {
		var allFieldCards = $gameTroop.fieldCards();
		var index = allFieldCards.indexOf(this._card);
		if (index > -1) {
			var win = SceneManager._scene._skillWindow;
			win._getEnemyField = true;
			var spritegroup = win.getSpritegroupFromName("field");
			spritegroup.popCardSpriteFromCard(-1, this._card);
		}
		var fieldCards = this._summoner.getZoneByName("field");
		index = fieldCards._data.indexOf(this._card);
		if (index > -1) {
			fieldCards.splice(index, 1);
		}

		index = $gameTroop._fieldCreatures.indexOf(this);
		if (index > -1)
			$gameTroop._fieldCreatures.splice(index, 1);

		BattleManager.exitZoneForcedActions(this._summoner, "field", this._card.id());
	}

	//SceneManager._scene._spriteset.removeCreature(this._card);
}

Game_CardCreature.prototype.getStatDisplays = function () {
	var dataEnemy = this.enemy();
	return DataManager.getStatDisplays(dataEnemy.id);
}

Game_CardCreature.prototype.isScaleRequested = function ()
{
	return !!this._scaleEffect;
}

Game_CardCreature.prototype.scaleEffect = function ()
{
	return this._scaleEffect;
}

Game_CardCreature.prototype.requestScale = function (scaleEffect)
{
	this._scaleEffect = scaleEffect;
}

Game_CardCreature.prototype.clearScaleEffect = function ()
{
	this._scaleEffect = null;
}

Game_CardCreature.prototype.performActionStart = function (action)
{
	this.requestScale(1.1);
	Game_Enemy.prototype.performActionStart.call(this, action);
}

Game_CardCreature.prototype.performActionEnd = function ()
{
	this.requestScale(1);
	Game_Enemy.prototype.performActionEnd.call(this);
}

Game_CardCreature.prototype.hasRemainingActions = function ()
{
	if (!this.canMoveThisTurn())
		return false;

	var fieldSkills = this._card.fieldSkills();
	for (var i = 0; i < fieldSkills.length; i++)
	{
		if (fieldSkills[i].uses < fieldSkills[i].limit && fieldSkills[i].limit != -1)
			return true;
	}

	return false;
}

Game_CardCreature.prototype.cardVar = function (index)
{
	var card = this._card;
	if (!card) return 0;
	return card.getVar(index);
}

Game_CardCreature.prototype.getAllTurnCardActions = function ()
{
	var enemyActions = $dataEnemies[this._enemyId]._turnActions;

	var stateActions = [];
	var states = this.states();
	for (var i = 0; i < states.length; i++)
	{
		if (states[i])
			stateActions[i] = states[i]._turnActions;
	}

	var totalActions = "";
	if (enemyActions && enemyActions != "")
	{
		if (totalActions == "")
			totalActions += enemyActions;
		else
			totalActions += "\n" + enemyActions;
	}
	for (var i = 0; i < stateActions.length; i++)
	{
		var sA = stateActions[i];
		if (sA && sA != "")
		{
			if (totalActions == "")
				totalActions += sA;
			else
				totalActions += "\n" + sA;
		}
	}
	return totalActions;

}


// Compatibility with YEP_AutoPassiveStates

//We don't want creatures gaining the passive states of enemies
Game_CardCreature.prototype.getPassiveStateData = function (obj)
{
	var array = Game_BattlerBase.prototype.getPassiveStateData.call(this, obj);
	if (array.length == 0)
		return array;

	var enemyArray = Yanfly.Param.APSEnemyPas;
	for (var i = 0; i < enemyArray.length; i++)
	{
		var state = enemyArray[i];
		var index = array.indexOf(state);
		if (index > -1)
			array.splice(index, 1);
	}

	return array;
};

// #endregion

//=============================================================================
// Sprite_Enemy
//=============================================================================
// #region

Myth.CGC.Summons.Sprite_Enemy_updateMain = Sprite_Enemy.prototype.updateMain;
Sprite_Enemy.prototype.updateMain = function ()
{
	Myth.CGC.Summons.Sprite_Enemy_updateMain.call(this);
	this.updateSelectability();
}

Sprite_Enemy.prototype.updateSelectability = function ()
{
	if (this._effectType != null) return;
	if (this._appeared == false) return;

	var highlightMode = this._battler._selectHighlightMode;
	var canSelect = true;
	if (highlightMode == 'targeting' && !this._battler.isEnabledForTargeting())
		canSelect = false;

	if (canSelect && this._appeared)
		this.opacity = 255;
	else
		this.opacity = 100;
}

// #endregion


//=============================================================================
// Sprite_CardCreature
//=============================================================================
// #region

function Sprite_CardCreature() {
	this.initialize.apply(this, arguments);
};

Sprite_CardCreature.prototype = Object.create(Sprite_Enemy.prototype);
Sprite_CardCreature.prototype.constructor = Sprite_CardCreature;

Sprite_CardCreature.prototype.initialize = function (battler) {
	Sprite_Enemy.prototype.initialize.call(this, battler);
	this.anchor.y = 0.5;
	this.createStatDisplays();

	this._targetScaleEffect = 0;
	this._scaleEffect = 0;
	this._scaleDuration = 0;
	
};

Sprite_CardCreature.prototype.createStatDisplays = function () {
	this._statDisplays = [];
	var creatureStats = this._enemy.getStatDisplays();
	for (var i = 0; i < creatureStats.length; i++) {
		this._statDisplays[i] = new Sprite_CreatureStat(creatureStats[i], this._enemy, true);
		this._statDisplays[i].x = creatureStats[i].coordinate.x;
		this._statDisplays[i].y = creatureStats[i].coordinate.y;
		this.addChild(this._statDisplays[i]);
	}
};

Sprite_CardCreature.prototype.initVisibility = function ()
{
	if (this._battler._isTransforming)
	{
		this._appeared = true;
	}
	else
	{
		this._appeared = false;
		this.opacity = 0;
		this.startEffect("appear");
	}
};

Sprite_CardCreature.prototype.setHome = function (x, y)
{
	var dataEnemy = this._enemy.enemy();
	this._homeX = x;
	this._homeY = y;
	var gridOffset = dataEnemy._gridOffset;
	if (gridOffset)
	{
		if (gridOffset.x)
			this._homeX += gridOffset.x;
		if (gridOffset.y)
			this._homeY += gridOffset.y;
	}
	this.updatePosition();
};

Sprite_CardCreature.prototype.updateCollapse = function () {
	Sprite_Enemy.prototype.updateCollapse.call(this);
	if (this.opacity <= 0) {
		SceneManager._scene._spriteset.removeCreature(this._enemy._card);
	}
}

Sprite_CardCreature.prototype.setupScale = function ()
{
	if (this._appeared && this._battler.isScaleRequested())
	{
		this.startScale(this._battler.scaleEffect());
		this._battler.clearScaleEffect();
	}
}

Sprite_CardCreature.prototype.startScale = function (scaleEffect)
{
	if (this.scale.x != scaleEffect)
	{
		this._targetScaleEffect = scaleEffect;
		this._scaleDuration = 10;
	}
}

Sprite_CardCreature.prototype.updateScale = function ()
{
	this.setupScale();
	if (this._scaleDuration > 0)
	{
		const d = this._scaleDuration;
		this.scale.x = (this.scale.x * (d - 1) + this._targetScaleEffect) / d;
		this.scale.y = this.scale.x;
		this._scaleDuration--;
/*		if (this._scaleDuration === 0)
		{
			this.onScaleEnd();
		}*/
	}
}

Sprite_CardCreature.prototype.updateMain = function ()
{
	Sprite_Enemy.prototype.updateMain.call(this);
	this.updateScale();
	//this.updateSelectability();
	this.updateDebugMode();
	this.updateCardTransform();
};

Sprite_CardCreature.prototype.updateSelectability = function ()
{
	if (this.scale.x != 1) return;
	if (this._scaleDuration > 0) //don't update while scaling
		return;

	var highlightMode = this._battler._selectHighlightMode;

	var canSelect = false;
	if (highlightMode == '')
	{
		if (this._battler.hasRemainingActions())
			canSelect = true;
		var tintAfterActions = Myth.CGC.Summons.partyCreatureSettings.tintAfterActions;
		if (this._battler.friendsUnit() == $gameTroop)
			tintAfterActions = Myth.CGC.Summons.troopCreatureSettings.tintAfterActions;
		if (!tintAfterActions)
			canSelect = true;
	}
	if (highlightMode == 'targeting' && this._battler.isEnabledForTargeting())
		canSelect = true;

	if (canSelect)
		this.setBlendColor([0, 0, 0, 0]);
	else
		this.setBlendColor([10, 10, 10, 128]);
}

Sprite_CardCreature.prototype.updateDebugMode = function ()
{
	if (this._debugMode != Myth.CGC.Summons.debugMode && this._battlerName != null)
	{
		this._debugMode = Myth.CGC.Summons.debugMode;
		if (this._debugDot == undefined)
		{
			var bitmap = new Bitmap(6, 6);
			bitmap.drawCircle(3, 3, 3, 'black')
			bitmap.drawCircle(3, 3, 2, 'white');
			this._debugDot = new Sprite(bitmap);
			this._debugDot.anchor.x = 0.5;
			this._debugDot.anchor.y = 0.5;
			this.addChild(this._debugDot);
		}

		this._debugDot.visible = this._debugMode;

	}
	if (this._debugDot && this._debugMode)
	{
		this._debugDot.scale.x = 1 / this.scale.x;
		this._debugDot.scale.y = 1 / this.scale.y;
	}
}

Sprite_CardCreature.prototype.updateCardTransform = function ()
{
	var targetScale = 1;
	if (this._battler._isTransforming)
	{
		var speed = 0.05;
		if (this._battler._newCard != null)
		{
			this.scale.x = this.scale.x - (speed);
			if (this.scale.x <= 0)
			{
				this.scale.x = 0;
				this.transformFieldCreature();
				this._battler._newCard = null;
			}
		}
		else
		{
			this.scale.x = this.scale.x + (speed);
			if (this.scale.x >= targetScale)
			{
				this.scale.x = targetScale;
				if (this._finishedTransformingTimer == undefined)
					this._finishedTransformingTimer = 10;
				this._finishedTransformingTimer--;
				if (this._finishedTransformingTimer == 0)
				{
					this._battler._isTransforming = false;
					this._finishedTransformingTimer = undefined;
				}

			}
		}
	}
}

Sprite_CardCreature.prototype.transformFieldCreature = function ()
{
	var newCreature = null;
	var newFieldCreatureId = this._battler._newFieldCreature;
	if (newFieldCreatureId == -1)
	{
		this.visible = false;
		var index = unit._fieldCreatures.indexOf(this._battler);
		if (index > -1)
		{
			unit._fieldCreatures.splice(index, 1);
		}
	}
	else
	{
		newCreature = new Game_CardCreature(newFieldCreatureId, this._battler.screenX(), this._battler.screenY(), this._battler._newCard, this._battler._summoner);

		var unit = this._battler.friendsUnit();
		var index = unit._fieldCreatures.indexOf(this._battler);
		if (index > -1)
		{
			unit._fieldCreatures.splice(index, 1, newCreature);
		}
	}

	this._battler._card._fieldCreature = newCreature;
	this._battler._newCard = null;
	this._battler._newFieldCreature = null;
	this.setBattler(newCreature);
	this._battler._isTransforming = true;
}

Sprite_CardCreature.prototype.updateStateSprite = function ()
{
	this._stateIconSprite.y = this.bitmap.height * 0.5 - Math.round((this.bitmap.height + 40) * 0.9);
	if (this._stateIconSprite.y < 20 - this.y)
	{
		this._stateIconSprite.y = 20 - this.y;
	}
};

// YEP_CoreEngine compatibility
if (Imported && Imported.YEP_CoreEngine)
{
	if (Yanfly.Param.ReposBattlers)
	{
		//undo the repositioning since it isn't necessary with Creatures.
		Sprite_CardCreature.prototype.setBattler = function (battler)
		{
			Yanfly.Core.Sprite_Enemy_setBattler.call(this, battler);
		}
	}
}


// #endregion

//=============================================================================
// Sprite_CreatureStat
//=============================================================================
// #region

function Sprite_CreatureStat() {
	this.initialize.apply(this, arguments);
};

Sprite_CreatureStat.prototype = Object.create(Sprite.prototype);
Sprite_CreatureStat.prototype.constructor = Sprite_CreatureStat;

Sprite_CreatureStat.prototype.initialize = function (creatureStat, battler, isSmooth) {
	this._creatureStat = creatureStat;
	this._battler = battler;
	this._statName = creatureStat.param;

	this._isSmooth = isSmooth;
	if (isSmooth == undefined)
		this._isSmooth = Myth.CGC.smoothCards;

	var bitmap = ImageManager.loadBitmap("img/CGC/summons/", creatureStat.backImage);
	Sprite.prototype.initialize.call(this, bitmap);
	this.bitmap.smooth = this._isSmooth;
	this.amount = NaN;
	this.originalAmount = battler[this._statName];
	this.anchor = new Point(0.5, 0.5);

	var textSpriteSize = 100;//Myth.CGC.fontSizes.creatureStat * 2;
	this._textSprite = new Sprite(new Bitmap(textSpriteSize, textSpriteSize));
	this._textSprite.bitmap.smooth = this._isSmooth;
	this._textSprite.anchor = new Point(0.5, 0.5);
	this.addChild(this._textSprite);
	//this.initializeTextSprite();
}

Sprite_CreatureStat.prototype.initializeTextSprite = function () {
	this.amount = NaN;
	this._textSprite.bitmap = new Bitmap(this.bitmap.width, this.bitmap.height);
	this._textSprite.bitmap.smooth = this._isSmooth;
}

Sprite_CreatureStat.prototype.update = function () {
	Sprite.prototype.update.call(this);
	this.updateParam();
}

Sprite_CreatureStat.prototype.battler = function ()
{
	return this._battler === 'number' ? $dataEnemies[this._battler] : this._battler;
}

Sprite_CreatureStat.prototype.updateParam = function ()
{
	var battler = this.battler();
	var value = battler[this._statName];
	if (value != this.amount) {
		this.amount = value;
		//if (this.amount == undefined) this.amount = "";
		var b = this._textSprite.bitmap;
		b.clear();
		this.setTextColor();
		b.fontSize = Myth.CGC.fontSizes.creatureStat;
		b.fontFace = Myth.CGC.Summons.creatureStatFont;
		var text = this.amount == undefined ? "" : this.amount;
		b.drawText(text, 0, 0, b.width, b.height, 'center');
	}
}

Sprite_CreatureStat.prototype.setTextColor = function () {
	var color = 'white';
	var value = this.amount;
	var originalValue = this.originalAmount;
	var battler = this.battler();
	var creature = battler;
	var textConditions = this._creatureStat.textConditions;
	if (!textConditions) {
		this._textSprite.textColor = color;
		return;
	}
	for (var i = 0; i < textConditions.length; i++) {
		try
		{
			var conditionMet = eval(textConditions[i].condition);
			if (conditionMet) {
				color = textConditions[i].textColor;
				break;
			}
		}
		catch (e) {
			console.warn(e);
			Myth.CGC.Summons.warnMessage(2, textConditions[i].textColor, textConditions[i].condition);

		}
	}
	this._textSprite.bitmap.textColor = color;
}

// #endregion

//=============================================================================
// Sprite_Animation
//=============================================================================
// #region

if (Myth.Util.usingMZ)
{
	Myth.CGC.Summons.Sprite_Animation_updateEffectGeometry = Sprite_Animation.prototype.updateEffectGeometry;
	Sprite_Animation.prototype.updateEffectGeometry = function ()
	{
		Myth.CGC.Summons.Sprite_Animation_updateEffectGeometry.call(this);
		var target = this._targets[0];
		if (target && target._enemy && target._enemy.isCreature())
		{
			var scale1 = this._animation.scale / 100;
			var scale2 = Myth.CGC.Summons.partyCreatureSettings.animationScale;
			if (target._enemy.friendsUnit() == $gameTroop)
				scale2 = Myth.CGC.Summons.troopCreatureSettings.animationScale;
			var finalScale = scale1 * scale2;
			if (this._handle)
			{
				this._handle.setScale(finalScale, finalScale, finalScale);
			}
		}

	}

	Myth.CGC.Summons.Sprite_Animation_targetSpritePosition = Sprite_Animation.prototype.targetSpritePosition;
	Sprite_Animation.prototype.targetSpritePosition = function (sprite)
	{
		//if (!sprite._battler.isCreature())
		var pos = Myth.CGC.Summons.Sprite_Animation_targetSpritePosition.call(this, sprite);

		if (sprite && sprite._battler && sprite._battler.isCreature())
		{
			pos.y += sprite.height / 2;
		}
		return pos;
	}
}
else
{
	Sprite_CardCreature.prototype.startAnimation = function (animation, mirror, delay)
	{
		Sprite_Enemy.prototype.startAnimation.call(this, animation, mirror, delay);
		if (animation.position == 3) return;

		var friends = this._battler.friendsUnit();
		var scale;
		if (friends == $gameParty)
			scale = Myth.CGC.Summons.partyCreatureSettings.animationScale;
		else
			scale = Myth.CGC.Summons.troopCreatureSettings.animationScale;
		var anim = this._animationSprites[this._animationSprites.length - 1];
		anim.scale.x = scale;
		anim.scale.y = scale;
	}

/*	Myth.CGC.Summons.Sprite_Animation_updateFrame = Sprite_Animation.prototype.updateFrame;
	Sprite_Animation.prototype.updateFrame = function ()
	{
		this.scale.x = Myth.CGC.Summons.creatureAnimationScale;
		this.scale.y = Myth.CGC.Summons.creatureAnimationScale;
		Myth.CGC.Summons.Sprite_Animation_updateFrame.call(this);
	}*/
}

// #endregion

//=============================================================================
// Sprite_StateIcon
//=============================================================================
// #region


Myth.CGC.Summons.Sprite_StateIcon_updateFrame = Sprite_StateIcon.prototype.updateFrame;
Sprite_StateIcon.prototype.updateFrame = function ()
{
	if (this._iconIndex == 0)
		this.setFrame(0, 0, 0, 0);
	else
		Myth.CGC.Summons.Sprite_StateIcon_updateFrame.call(this);
}

// #endregion

//=============================================================================
// Spritegroup_Field
//=============================================================================
// #region

function Spritegroup_Field()
{
	this.initialize.apply(this, arguments);
}

Spritegroup_Field.prototype.initialize = function ()
{
	this.cardSprites = [];
	this.cardOwners = [];
	this.cardPositions = [];
	this._currentBattler = null;
}

Spritegroup_Field.prototype.cardSprite = function (index)
{
	return this.cardSprites[index];
}

Spritegroup_Field.prototype.cardSpriteAtPosition = function (index)
{
	for (var i = 0; i < this.cardSprites.length; i++)
	{
		if (this.cardPositions[i] == index)
			return this.cardSprites[i];
	}
	return null;
	//return this.cardSprites[index];
}

Spritegroup_Field.prototype.cardSpriteFromCard = function (card)
{
	for (var i = 0; i < this.cardSprites.length; i++)
	{
		if (this.cardSprites[i].card() == card)
			return this.cardSprites[i];
	}
	return null;
}

Spritegroup_Field.prototype.cardOwner = function (index)
{
	if (index instanceof Game_Card)
	{
		for (var i = 0; i < this.cardSprites.length; i++)
		{
			if (this.cardSprites[i].card == index)
				return this.cardOwners[i];
		}
	}
	else
		return this.cardOwners[index];
}

Spritegroup_Field.prototype.cardOwnerAtPosition = function (index)
{
	for (var i = 0; i < this.cardOwners.length; i++)
	{
		if (this.cardPositions[i] == index)
			return this.cardOwners[i];
	}
	return null;
}

Spritegroup_Field.prototype.getCardSprites = function ()
{
	return this.cardSprites;
}
Spritegroup_Field.prototype.getBattlerCardSprites = function (battler)
{
	if (!battler) battler = this._currentBattler;
	if (!battler) return -1;
	var sprites = [];
	for (i in this.cardSprites)
	{
		var cardSprite = this.cardSprite(i);
		var owner = this.cardOwner(i);
		if (owner == battler)
			sprites.push(cardSprite);
	}
	return sprites;
}

Spritegroup_Field.prototype.pushCardSprite = function (sprite, battler, index)
{
	if (!battler)
		battler = this._currentBattler;
	if (!battler)
		battler = -1;

	if (!index)
		index = this.cardSprites.length;


	this.cardSprites[index] = sprite;
	this.cardOwners[index] = battler;
	this.cardPositions[index] = this.findFirstEmptyIndex();
}

Spritegroup_Field.prototype.findFirstEmptyIndex = function ()
{
	var positions = this.cardPositions.slice();
	positions.sort((a, b) => a - b);
	for (var i = 0; i < positions.length; i++)
	{
		if (positions[i] != i)
			return i;
	}

	return this.cardSprites.length - 1;
}

Spritegroup_Field.prototype.popCardSprite = function (actor, index)
{
	var card = this.cardSprites[index];
	this.cardSprites.splice(index, 1);
	this.cardOwners.splice(index, 1);
	this.cardPositions.splice(index, 1);
	return card;
}

Spritegroup_Field.prototype.popCardSpriteFromCard = function (actor, card)
{
	for (var i = 0; i < this.cardSprites.length; i++)
	{
		if (this.cardSprites[i].card() == card)
			return this.popCardSprite(-1, i);
	}
	return null;
}

Spritegroup_Field.prototype.isSlotEmpty = function (index)
{
	return this.cardSprites[index] == null;
}

Spritegroup_Field.prototype.getIndexOfCard = function (card, actor)
{
	for (var i = 0; i < this.cardSprites.length; i++)
	{
		if (this.cardSprites[i].card() == card)
			return i;
	}
	return -1;
}

// #endregion


//=============================================================================
// Window_BattleSkill
//=============================================================================
// #region

Myth.CGC.Summons.Window_BattleSkill_getSpritegroupFromName = Window_BattleSkill.prototype.getSpritegroupFromName;
Window_BattleSkill.prototype.getSpritegroupFromName = function (name)
{
	if (name == 'field' && $gameSystem._cardGridEnabled)
	{
		if (this._getEnemyField == true) {
			this._getEnemyField = false;
			return SceneManager._scene._spriteset._troopFieldSpritegroup;
		}
		return SceneManager._scene._spriteset._partyFieldSpritegroup;
	}
	else
		return Myth.CGC.Summons.Window_BattleSkill_getSpritegroupFromName.call(this, name);
}

Myth.CGC.Summons.Window_BattleSkill_resolveUsedCard = Window_BattleSkill.prototype.resolveUsedCard;
Window_BattleSkill.prototype.resolveUsedCard = function (zoneName) {
	if (this._activeCardSprite && zoneName == 'field' && $gameSystem._cardGridEnabled) {
		var card = this._activeCardSprite.card();
		card.resetAllFieldSkillUses();
		BattleManager.lastPlayedCard = card;
		var destinationSpritegroup = this.getSpritegroupFromName(zoneName);
		//destination.add(card);
		//SoundManager.playEnterZone(zoneName);
		destinationSpritegroup.pushCardSprite(this._activeCardSprite, this._actor);

		this._activeCardSprite = null;
	}

	Myth.CGC.Summons.Window_BattleSkill_resolveUsedCard.call(this, zoneName);
};

Myth.CGC.Summons.Window_BattleSkill_cursorUp = Window_BattleSkill.prototype.cursorUp;
Window_BattleSkill.prototype.cursorUp = function (wrap)
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.Window_BattleSkill_cursorUp.call(this, wrap);

	Input.clear();
	SceneManager._scene.activateFieldWindow();
	this.deselect();
};

// #endregion


//=============================================================================
// Window_BattleField
//=============================================================================
// #region

//following the naming conventions of RPG Maker, this does make it funny
function Window_BattleField() {
	this.initialize.apply(this, arguments);
};

Window_BattleField.prototype = Object.create(Window_Selectable.prototype);
Window_BattleField.prototype.constructor = Window_BattleField;

Window_BattleField.prototype.initialize = function (spriteset)
{
	var partySprite = spriteset._partyGridSprite;
	var troopSprite = spriteset._troopGridSprite;
	var partySpritegroup = spriteset._partyFieldSpritegroup;
	var troopSpritegroup = spriteset._troopFieldSpritegroup;

	this._gridSprite = null;
	this._fieldSpritegroup = null;
	this._mode = '';

	if (Myth.Util.usingMZ) {
		Window_Selectable.prototype.initialize.call(this, new Rectangle(0, 0, 0, 0));
	}
	else {
		Window_Selectable.prototype.initialize.call(this, 0, 0, 0, 0);
	}

	//this.hide();

	this._partyGridSprite = partySprite;
	this._troopGridSprite = troopSprite;

	this._partySpritegroup = partySpritegroup;
	this._troopSpritegroup = troopSpritegroup;

	this._playerWindow = null;
	this._enemyWindow = null;

	this._playerCard = null;

	this.initializeFieldMatrix();
};

Window_BattleField.prototype.initializeFieldMatrix = function()
{
	var troopFieldPos = Myth.CGC.Summons.coordinates.troopFieldSettings.fieldPosition;
	var partyFieldPos = Myth.CGC.Summons.coordinates.partyFieldSettings.fieldPosition;

	var diffX = (troopFieldPos.x - partyFieldPos.x);
	var diffY = (troopFieldPos.y - partyFieldPos.y);

	var isVertical = true;
	if (Math.abs(diffX) > Math.abs(diffY))
		isVertical = false;

	// if diffY is negative it means that the party field is below the troop field
	// if diffX is negative it means the party field is to the right of the troop field

	var matrix = [
		['party', 'troop'],
		['troop', 'party']
	];
	if (isVertical)
	{ 
		if (diffY < 0)
		{
			matrix = [
				['troop'],
				['party']
			];
		}
		else
		{
			matrix = [
				['party'],
				['troop']
			]
		}
	}
	else
	{
		if (diffX < 0)
		{
			matrix = [
				['troop', 'party']
			];
		}
		else
		{
			matrix = [
				['party', 'troop']
			]
		}
	}

	this._fieldMatrix = matrix;
	this._matrixPosition = {x: -1, y:-1};
}

Window_BattleField.prototype.updateArrows = function() {
    this.downArrowVisible = false;
    this.upArrowVisible = false;;
};

Window_BattleField.prototype.setSpritegroups = function (partySpritegroup, enemySpritegroup)
{
	this._partySpritegroup = partySpritegroup;
	this._troopSpritegroup = enemySpritegroup;
}

Window_BattleField.prototype.setStatusWindows = function (playerWindow, enemyWindow)
{
	this._playerWindow = playerWindow;
	this._enemyWindow = enemyWindow;
}

Window_BattleField.prototype.setField = function (side)
{
	this._side = side;
	switch (side)
	{
		case "party":
			this._gridSprite = this._partyGridSprite;
			this._fieldSpritegroup = this._partySpritegroup;
			break;
		case "troop":
			this._gridSprite = this._troopGridSprite;
			this._fieldSpritegroup = this._troopSpritegroup;
			break;
		default:
			this._gridSprite = null;
			this._fieldSpritegroup = null;
	}
	this.findMatrixPosition();
}

Window_BattleField.prototype.findMatrixPosition = function()
{
	this._matrixPosition.x = -1;
	this._matrixPosition.y = -1;

	for (var i = 0; i < this._fieldMatrix.length; i++) // y dimension
	{
		for (var j = 0; j < this._fieldMatrix[i].length; j++) // x dimension
		{
			if (this._fieldMatrix[i][j] == this._side)
			{
				this._matrixPosition.x = j;
				this._matrixPosition.y = i;
			}
		}
	}
}

Window_BattleField.prototype.setMode = function (mode) {
	this._mode = mode;
}

Window_BattleField.prototype.mode = function () {
	return this._mode;
}

Window_BattleField.prototype.maxCols = function () {
	return Myth.CGC.Summons.gridWidth();
};

Window_BattleField.prototype.numVisibleRows = function () {
	return this.maxRows();
};

Window_BattleField.prototype.maxRows = function () {
	return Myth.CGC.Summons.gridHeight();
};

Window_BattleField.prototype.maxItems = function () {
	return Myth.CGC.Summons.gridWidth() * Myth.CGC.Summons.gridHeight();
};

Window_BattleField.prototype.select = function (index) {
	Window_Selectable.prototype.select.call(this, index);
	if (this._gridSprite)
		this._gridSprite.select(index);
};

Window_BattleField.prototype.deselect = function () {
	Window_Selectable.prototype.deselect.call(this);
	if (this._partyGridSprite)
		this._partyGridSprite.select(-1);
	if (this._troopGridSprite)
		this._troopGridSprite.select(-1);
}

Window_BattleField.prototype.toggleSelectionHighlight = function () {
	if (!this._fieldSpritegroup)
		return;

	var cardSprites = this._fieldSpritegroup.getCardSprites();
	var selectedCards = SceneManager._scene._cardSelectionWindow.selectedCards();
	for (var i = 0; i < cardSprites.length; i++) {
		var card = cardSprites[i].card();
		card.reselectEffect();
		var isInSelected = selectedCards.indexOf(card) > - 1 ? true : false;
		if (isInSelected)
			card.selectEffect();
		else
			card.deselectEffect();
	}
}

Window_BattleField.prototype.endSelectionHighlight = function () {
	if (!this._fieldSpritegroup)
		return;

	var cardSprites = this._fieldSpritegroup.getCardSprites();
	for (var i = 0; i < cardSprites.length; i++) {
		var card = cardSprites[i].card();
		card.deselectEffect();
	}
}

Window_BattleField.prototype.updateHelp = function () {
	this.setHelpWindowItem(this.item());
};

Window_BattleField.prototype.item = function () {
	var card = this.card();
	if (!card) return null;
	var skill = $dataSkills[card.id()];

	return skill;
};

Window_BattleField.prototype.card = function () {
	if (!this._fieldSpritegroup) return null;

	var cardSprite = this._fieldSpritegroup.cardSpriteAtPosition(this.index());
	if (!cardSprite) return null;
	var card = cardSprite.card();
	if (!card) return null;

	return card;
};

Window_BattleField.prototype.getPlayerCard = function () {
	if (this._playerCard == null) {
		var card = this.card();
		if (card)
			this._playerCard = card;
	}
	return this._playerCard;
};

Window_BattleField.prototype.clearPlayerCard = function () {
	this._playerCard = null;
}

Window_BattleField.prototype.owner = function () {
	if (!this._fieldSpritegroup) return null;
	var owner = this._fieldSpritegroup.cardOwnerAtPosition(this.index());
	if (owner == -1) return null;

	return owner;
}

Window_BattleField.prototype.isCurrentItemEnabled = function () {
	var mode = this._mode;
	var item = this.item();
	var card = this.card();
	if (!item)
		return false;
	if (mode == '' && this._side == 'troop')
		return false;
	if (item._fieldSkills && item._fieldSkills.length > 0 && mode == '')
		return true;
	if (item._fieldCreature)
	{
		if (mode == '')
			return true;
		else if (mode == 'targeting')
			return card._fieldCreature.isEnabledForTargeting();
	}
	if (item._fieldCreature && (mode == 'targeting' || mode == ''))
		return true;
	if (mode == 'selecting') {
		if (BattleManager._cardTypeToSelect) {
			if (DataManager.isSkillOfType(item.id, BattleManager._cardTypeToSelect))
				return true;
			return false;
		}
		return true;
	}
	return false;
};

Window_BattleField.prototype.cursorDown = function (wrap)
{
	var index = this.index();
	var maxItems = this.maxItems();
	var maxCols = this.maxCols();
	if (index < maxItems - maxCols || (wrap && maxCols === 1)) {
		this.select((index + maxCols) % maxItems);
	}
	else if (this._mode == '')
	{
		if (this._fieldMatrix.length - 1 > this._matrixPosition.y)
		{
			var newIndex = (index % maxCols);
			this._matrixPosition.y++;
			var newField = this._fieldMatrix[this._matrixPosition.y][this._matrixPosition.x];
			this.deselect();
			this.setField(newField);
			this.select(newIndex);
		}
		else
			SceneManager._scene.deactivateFieldWindow();
	}
			
};

Window_BattleField.prototype.cursorLeft = function (wrap) {
	var index = this.index();
	var maxItems = this.maxItems();
	var maxCols = this.maxCols();
	if (index % maxCols > 0)
		this.select((index - 1 + maxItems) % maxItems);
	else if (this._mode == '' && this._fieldMatrix[0].length > 1 && this._matrixPosition.x > 0)
	{
		this._matrixPosition.x--;
		var newField = this._fieldMatrix[this._matrixPosition.y][this._matrixPosition.x];
		var newIndex = index + maxCols - 1;
		this.deselect();
		this.setField(newField);
		this.select(newIndex);
	}
}

Window_BattleField.prototype.cursorRight = function (wrap) {
	var index = this.index();
	var maxItems = this.maxItems();
	var maxCols = this.maxCols();
	if (index % maxCols < (maxCols - 1))
		this.select((index + 1) % maxItems);
	else if (this._mode == '' && this._fieldMatrix[0].length - 1 > this._matrixPosition.x)
	{
		this._matrixPosition.x++;
		var newField = this._fieldMatrix[this._matrixPosition.y][this._matrixPosition.x];
		var newIndex = index - maxCols + 1;
		this.deselect();
		this.setField(newField);
		this.select(newIndex);
	}
}

Window_BattleField.prototype.cursorUp = function (wrap) {
	var index = this.index();
	var maxItems = this.maxItems();
	var maxCols = this.maxCols();
	if (index >= maxCols || (wrap && maxCols === 1)) {
		this.select((index - maxCols + maxItems) % maxItems);
	}
	else
	{
		var troopToMoveTo = this.canMoveToUnitWindow();
		if (troopToMoveTo)
		{
			if (troopToMoveTo == "party")
				Myth.CGC.Summons.Scene_Battle_selectActorSelection.call(SceneManager._scene);
			else if (troopToMoveTo == "troop")
			{
				SceneManager._scene.startEnemyWindowSelection();
			}
			this.deactivate();
			this.select(-1);
			this._helpWindow.hide();
		}
		else if (this._mode == '' && this._fieldMatrix.length > 1 && this._matrixPosition.y > 0)
		{
			this._matrixPosition.y--;
			var newField = this._fieldMatrix[this._matrixPosition.y][this._matrixPosition.x];
			var newIndex = maxItems - (maxCols - (index % maxItems));
			this.deselect();
			this.setField(newField);
			this.select(newIndex);
		}
	}
};

Window_BattleField.prototype.canMoveToUnitWindow = function ()
{
	var canMove = this._mode == "targeting" && BattleManager.fieldTargetMode != "creature";
	if (!canMove) return false;
	return this._side;
}

Window_BattleField.prototype.giveCardToRemovedSprites = function (card) {

	var cardSprite = this._fieldSpritegroup.cardSpriteFromCard(card);

	var trueX = cardSprite.x + this.x;
	var trueY = cardSprite.y + this.y;
	//cardSprite._enabledSprite.hide();
	//cardSprite.scale = this.cardScale();

	var skillWindow = SceneManager._scene._skillWindow;
	skillWindow._removedSprites.getCardSprites().push(cardSprite);

	this._fieldSpritegroup.popCardSpriteFromCard(-1, card);
	//this._cardSprites.splice(index, 1);
	cardSprite.x = trueX;
	cardSprite.y = trueY;

	//this._fieldSpritegroup.removeChild(cardSprite);
	SceneManager._scene.addCardSprite(cardSprite);

	var creature = cardSprite.card()._fieldCreature;
	if (creature) {
		var owner = creature._summoner;
		owner.removeCreature(cardSprite.card());
		cardSprite.show();
	}
}

// #region Touch input

Window_BattleField.prototype.isTouchedInsideFrame = function ()
{
	return true;
}


if (!Myth.Util.usingMZ)
{
	Window_BattleField.prototype.hitIndex = function ()
	{
		const touchPos = new Point(TouchInput.x, TouchInput.y);
		const localPos = this.worldTransform.applyInverse(touchPos);
		return this.hitTest(localPos.x, localPos.y);
	};

	Window_BattleField.prototype.onTouch = function(triggered) {
    var lastIndex = this.index();
    var x = this.canvasToLocalX(TouchInput.x);
    var y = this.canvasToLocalY(TouchInput.y);
    var hit = this.hitIndexEitherGrid(x, y);
    if (hit.index >= 0) {
		if (hit.side == this._side)
		{
			if (hit.index === this.index()) {
				if (triggered && this.isTouchOkEnabled()) {
					this.processOk();
				}
			} else if (this.isCursorMovable()) {
				this.select(hit.index);
			}
		}
        else if (this._mode == '')
		{
			this.deselect();
			this.setField(hit.side);
			TouchInput.clear();
			this.select(hit.index);
			
		}
    } else if (this._stayCount >= 10) {
        if (y < this.padding) {
            this.cursorUp();
        } else if (y >= this.height - this.padding && this._mode == '') {
			this.processCancel();
			if (!this.active)
			{
				SceneManager._scene._skillWindow.activate();
				SceneManager._scene._skillWindow.onTouch();
			}
        }
    }
    if (this.index() !== lastIndex) {
        SoundManager.playCursor();
    }
};
}
else
{
	Window_BattleField.prototype.onTouchSelect = function (trigger)
	{
		this._doubleTouch = false;
		if (this.isCursorMovable())
		{
			const lastIndex = this.index();
			const hit = this.hitIndexEitherGrid();
			if (hit.index >= 0 && hit.side == this._side)
			{
				if (hit.index === this.index())
				{
					this._doubleTouch = true;
				}
				this.select(hit.index);
			}
			if (trigger && this.index() !== lastIndex)
			{
				this.playCursorSound();
			}
		}
	};

	Window_BattleField.prototype.onTouchOk = function ()
	{
		if (this.isTouchOkEnabled())
		{
			const hitIndex = this.hitIndex();
			if (this._cursorFixed)
			{
				if (hitIndex === this.index())
				{
					this.processOk();
				}
			} else if (hitIndex >= 0)
			{
				this.processOk();
			}
		}
		else if (this._mode == '')
		{
			this.processCancel();
			if (!this.active)
			{
				SceneManager._scene._skillWindow.activate();
				SceneManager._scene._skillWindow.selectInactiveSelection();
			}

		}
	};
}

Window_BattleField.prototype.hitIndexEitherGrid = function()
{
	const touchPos = new Point(TouchInput.x, TouchInput.y);
	const localPos = this.worldTransform.applyInverse(touchPos);
	var partyIndex = this.hitTestGrid(localPos.x, localPos.y, this._partyGridSprite);
	var troopIndex = this.hitTestGrid(localPos.x, localPos.y, this._troopGridSprite);

	if (partyIndex > -1)
		return { side: "party", index: partyIndex};
	if (troopIndex > -1)
		return {side: "troop", index: troopIndex};

	return {side: "", index: -1};
}

Window_BattleField.prototype.hitTestGrid = function(x, y, grid)
{
	var gridWidth = Myth.CGC.Summons.gridWidth();
	var tileWidth = Myth.CGC.Summons.tileWidth();
	var gridHeight = Myth.CGC.Summons.gridHeight();
	var tileHeight = Myth.CGC.Summons.tileHeight();

	var rect = new Rectangle(grid.x + grid._fieldSettings.firstTileOffset.x, grid.y + grid._fieldSettings.firstTileOffset.y, 0, 0);
	rect.width = gridWidth * tileWidth;
	rect.height = gridHeight * tileHeight;
	if (rect.contains(x, y))
	{
		const cx = x - rect.x;
		const cy = y - rect.y;
		var indexX = Math.floor(cx / tileWidth);
		var indexY = Math.floor(cy / tileHeight);
		var index = (indexX) + (indexY * gridWidth); // avoids the *0 problem
		return index;
	}
	return -1;
}

Window_BattleField.prototype.hitTest = function (x, y)
{
	var grid = this._gridSprite || this._partyGridSprite;

	var gridWidth = Myth.CGC.Summons.gridWidth();
	var tileWidth = Myth.CGC.Summons.tileWidth();
	var gridHeight = Myth.CGC.Summons.gridHeight();
	var tileHeight = Myth.CGC.Summons.tileHeight();

	var rect = new Rectangle(grid.x + grid._fieldSettings.firstTileOffset.x, grid.y + grid._fieldSettings.firstTileOffset.y, 0, 0);
	rect.width = gridWidth * tileWidth;
	rect.height = gridHeight * tileHeight;
	if (rect.contains(x, y))
	{
		const cx = x - rect.x;
		const cy = y - rect.y;
		var indexX = Math.floor(cx / tileWidth);
		var indexY = Math.floor(cy / tileHeight);
		var index = (indexX) + (indexY * gridWidth); // avoids the *0 problem
		return index;
	}
	return -1;
};

Window_BattleField.prototype.processTouch = function ()
{
	if (this.canTouchToActivate())
	{
		var hit = this.hitIndexEitherGrid();
		if (this._playerWindow.active)
		{
			SceneManager._scene.activateFieldWindow('party', 'targeting');
			this._playerWindow.deactivate();
			this._playerWindow.deselect();
		}
		else if (this._enemyWindow.active)
		{
			SceneManager._scene.activateFieldWindow('troop', 'targeting');
			this._enemyWindow.deactivate();
			this._enemyWindow.select(-1);
			this._enemyWindow.hide();
		}
		else
		{
			SceneManager._scene.activateFieldWindow(hit.side);
		}
			

		SceneManager._scene._skillWindow.deselect();
		TouchInput._clicked = false;
		if (Myth.Util.usingMZ)
			this.onTouchSelect(false);
	}
	Window_Selectable.prototype.processTouch.call(this);
}

Window_BattleField.prototype.canTouchToActivate = function ()
{
	if (!this.isOpen()) return false;
	if (!this.visible) return false;
	if (this.active) return false;

	var skillWindowOpen = SceneManager._scene._skillWindow.isOpenAndActive();
	if (this._mode == '' && !skillWindowOpen) return false;
	//if (!skillWindowOpen) return false;
	//if (this._mode == "targeting" && !this.canMoveToUnitWindow()) return false;
	if (this._playerWindow.active || this._enemyWindow.active)
		if (BattleManager.fieldTargetMode == 'battler') return false;
	
	//if (this._side == null) return false;
	if (Myth.Util.usingMZ)
	{
		var hit = this.hitIndexEitherGrid();
		if (hit.index < 0) return false;
		if (!this.isHoverEnabled()) return false;
		if (!TouchInput.isClicked()) return false;
		if (BattleManager._phase != 'input') return false;
	}
	else
	{
		var hit = this.hitIndexEitherGrid();
		if (hit.index < 0) return false;
		if (!TouchInput.isTriggered()) return false;
	}

	return true;
}

// #endregion

// #endregion


//=============================================================================
// Window_BattleFieldSkill
//=============================================================================
// #region

function Window_BattleFieldSkill() {
	this.initialize.apply(this, arguments);
};

Window_BattleFieldSkill.prototype = Object.create(Window_SkillList.prototype);
Window_BattleFieldSkill.prototype.constructor = Window_BattleFieldSkill;

Window_BattleFieldSkill.prototype.initialize = function () {

	var coords = Myth.CGC.Summons.coordinates;
	this._card = null;
	this._fieldSkills = [];
	var width = coords.fieldSkillWidth;
	var height = this.fittingHeight(coords.fieldSkillRows);
	var x = coords.fieldSkillWindow.x; //Graphics.boxWidth - width - 16;
	var y = coords.fieldSkillWindow.y;
	//var y = Graphics.boxHeight / 2 - (height / 2);

	if (Myth.Util.usingMZ) {
		Window_SkillList.prototype.initialize.call(this, new Rectangle(x, y, width, height));
	}
	else {
		Window_SkillList.prototype.initialize.call(this, x, y, width, height);
	}

	this.openness = 0;
}

Window_BattleFieldSkill.prototype.setCard = function (card, battler) {
	this._card = card;
	this._fieldSkills = [];
	this._actor = battler;
	this._trueBattler = battler;
	if (card) {
		var dataSkill = $dataSkills[card.id()];
		this._fieldSkills = card._fieldSkills;
		if (dataSkill._fieldCreature) {
			var dataEnemy = $dataEnemies[dataSkill._fieldCreature];
			if (this._fieldSkills == undefined) {
				this._fieldSkills = [];
				for (var i = 0; i < dataEnemy.actions.length; i++) {
					var fieldSkill = {
						skillId: dataEnemy.actions[i].skillId,
						limit: -1,
						limitType: "",
						uses: 0
					}
					this._fieldSkills[i] = fieldSkill;
				}
			}


			this._trueBattler = card._fieldCreature;
		}
	}
	//this.updatePosition();
	this.refresh();
};

Window_BattleFieldSkill.prototype.updatePosition = function (x, y)
{
	var coords = Myth.CGC.Summons.coordinates;
	if (coords.followFieldSkill)
	{
		var offsetX = coords.followFieldSkillOffset.x;
		var offsetY = coords.followFieldSkillOffset.y;
		this.x = x + offsetX;
		this.y = y + offsetY;
	}
}

/*Window_BattleFieldSkill.prototype.getCard = function (index)
{
	if (!this._actor) return null;
	var card = this._card
	return card;
};*/

Window_BattleFieldSkill.prototype.makeItemList = function () {
	this._data = [];
	if (this._fieldSkills && this._fieldSkills.length > 0) {
		for (var i = 0; i < this._fieldSkills.length; i++) {
			var skill = $dataSkills[this._fieldSkills[i].skillId];
			this._data[i] = skill;
		}
	} else {
		this._data = [];
	}
};

Window_BattleFieldSkill.prototype.drawItem = function (index) {
	Window_SkillList.prototype.drawItem.call(this, index);
	var skill = this._data[index];
	var fieldSkill = this._fieldSkills[index];
	if (fieldSkill && fieldSkill.limitType != "") {
		var costWidth = this.costWidth();
		var rect = this.itemRect(index);
		rect.width -= Myth.Util.usingMZ ? this.itemPadding() : this.textPadding();
		rect.width -= costWidth;
		this.changePaintOpacity(this.isEnabled(skill));
		this.resetTextColor();

		var remaining = fieldSkill.limit - fieldSkill.uses;
		this.drawText(remaining + "/" + fieldSkill.limit, rect.x, rect.y, rect.width / 3 * 2, 'right');
		this.changePaintOpacity(1);
	}
}

Window_BattleFieldSkill.prototype.maxCols = function () {
	return 1;
};

Window_BattleFieldSkill.prototype.maxItems = function () {
	return this._data ? this._data.length : 0;
};

Window_BattleFieldSkill.prototype.isEnabled = function (item) {

	var canUse = false;
	if (this._trueBattler)
		canUse = this._trueBattler.canUse(item);
	else
		canUse = this._actor && this._actor.canUse(item);

	if (canUse) {
		var index = this._data.indexOf(item);
		var fieldSkill = this._fieldSkills[index];
		if (fieldSkill.limitType != "")
			canUse = fieldSkill.uses < fieldSkill.limit;
	}
	return canUse;
};

Window_BattleFieldSkill.prototype.onTouch = function (triggered) {
	Myth.CGC.Window_BattleSkill_onTouch.call(this, triggered);
}

Window_BattleFieldSkill.prototype.select = function (index)
{
	var card = this._card;
	if (card && this._helpWindow)
	{
		this._helpWindow.setCard(card);
	}
	if (Myth.CGC.ICV)
		Myth.CGC.ICV.Window_SkillList_select.call(this, index);
	else
		Window_SkillList.prototype.select.call(this, index);
}

// #endregion


//=============================================================================
// Window_BattleEnemy
//=============================================================================
// #region

Myth.CGC.Summons.Window_BattleEnemy_cursorDown = Window_BattleEnemy.prototype.cursorDown;
Window_BattleEnemy.prototype.cursorDown = function (wrap)
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.Window_BattleEnemy_cursorDown.call(this, wrap);

	var index = this.index();
	var maxItems = this.maxItems();
	var maxCols = this.maxCols();
	var canTargetField = (BattleManager.fieldTargetMode != "battler");
	if (index + maxCols >= maxItems && canTargetField)
	{
		Input.clear();
		SceneManager._scene.activateFieldWindow('troop', 'targeting');
		this.deactivate();
		this.select(-1);
		this.hide();

/*		SceneManager._scene._enemyStatusWindow.select(-1);
		SceneManager._scene._enemyStatusWindow.deactivate();*/
	}
	else {
		this.select((index + maxCols) % maxItems);
	}
}

Window_BattleEnemy.prototype.deselect = function ()
{
	Window_Selectable.prototype.deselect.call(this);
	if ($gameSystem._cardGridEnabled && this._enemyStatusWindow)
	{
		this._enemyStatusWindow.deactivate();
		this._enemyStatusWindow.deselect();
	}
}

Window_BattleEnemy.prototype.deactivate = function ()
{
	Window_Selectable.prototype.deactivate.call(this);
	if ($gameSystem._cardGridEnabled && this._enemyStatusWindow)
	{
		this._enemyStatusWindow.deselect();
		this._enemyStatusWindow.deactivate();
	}
}

Myth.CGC.Summons.Window_BattleEnemy_isCurrentItemEnabled = Window_BattleEnemy.prototype.isCurrentItemEnabled
Window_BattleEnemy.prototype.isCurrentItemEnabled = function ()
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.Window_BattleEnemy_isCurrentItemEnabled.call(this);

	if (this._enemyStatusWindow)
		return this._enemyStatusWindow.isCurrentItemEnabled();
	return false;
}

// #endregion


//=============================================================================
// Window_BattleActor
//=============================================================================
// #region

Myth.CGC.Summons.Window_BattleActor_cursorDown = Window_BattleActor.prototype.cursorDown;
Window_BattleActor.prototype.cursorDown = function (wrap)
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.Window_BattleActor_cursorDown.call(this, wrap);

	var index = this.index();
	var maxItems = this.maxItems();
	var maxCols = this.maxCols();
	var canTargetField = (BattleManager.fieldTargetMode != "battler");
	if (index + maxCols >= maxItems && canTargetField) {
		SceneManager._scene.activateFieldWindow('party', 'targeting');
		this.deactivate();
		this.select(-1);
		this.hide();
	}
	else {
		this.select((index + maxCols) % maxItems);
	}
}

Window_BattleActor.prototype.activate = function ()
{
	Window_Selectable.prototype.activate.call(this);
	if (!$gameSystem._cardGridEnabled) return;

	var statusWindow = SceneManager._scene._statusWindow;
	if (statusWindow)
		statusWindow.activate();
}

Window_BattleActor.prototype.deactivate = function ()
{
	Window_Selectable.prototype.deactivate.call(this);
	if (!$gameSystem._cardGridEnabled) return;

	var statusWindow = SceneManager._scene._statusWindow;
	if (statusWindow)
		statusWindow.deactivate();
}

Myth.CGC.Summons.Window_BattleActor_windowWidth = Window_BattleActor.prototype.windowWidth;
Window_BattleActor.prototype.windowWidth = function ()
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.Window_BattleActor_windowWidth.call(this);
	return 0;
};

Myth.CGC.Summons.Window_BattleActor_windowHeight = Window_BattleActor.prototype.windowHeight;
Window_BattleActor.prototype.windowHeight = function ()
{
	if (!$gameSystem._cardGridEnabled)
		return Myth.CGC.Summons.Window_BattleActor_windowHeight.call(this);
	return 0;
};

// #endregion

//=============================================================================
// Window_ActorStatus
//=============================================================================
// #region

Window_ActorStatus.prototype.updateCursor = function ()
{
	if (this.active)
		this.setCursorRect(0, 0, this.contents.width, this.contents.height);
	else
		this.setCursorRect(0, 0, 0, 0);
};

// #region touch input

Window_ActorStatus.prototype.processTouch = function ()
{
	if (this.canTouchToActivate())
	{
		Myth.CGC.Summons.Scene_Battle_selectActorSelection.call(SceneManager._scene);
		SceneManager._scene._fieldWindow.deactivate();
		SceneManager._scene._fieldWindow.select(-1);
		TouchInput.clear();
	}
	Window_BattleStatus.prototype.processTouch.call(this);
}

Window_ActorStatus.prototype.canTouchToActivate = function ()
{
	if (!$gameSystem._cardGridEnabled) return false;
	if (!this.isOpen()) return false;
	if (!this.visible) return false;
	if (this.active) return false;
	if (SceneManager._scene._fieldWindow.canMoveToUnitWindow() != "party") return false;

	if (Myth.Util.usingMZ)
	{
		if (this.hitIndex() < 0) return false;
		if (!this.isHoverEnabled()) return false;
		if (!TouchInput.isClicked()) return false;
	}
	else
	{
		if (!this.isTouchedInsideFrame()) return false;
		if (!TouchInput.isTriggered()) return false;
	}

	return true;
}

if (!Myth.Util.usingMZ)
{
	Window_ActorStatus.prototype.hitIndex = function ()
	{
		const touchPos = new Point(TouchInput.x, TouchInput.y);
		const localPos = this.worldTransform.applyInverse(touchPos);
		return this.hitTest(localPos.x, localPos.y);
	};
}

// #endregion

// #endregion

//=============================================================================
// Window_PartyStatus
//=============================================================================
// #region

Window_PartyStatus.prototype.processTouch = function ()
{
	return;
}

// #endregion

//=============================================================================
// Window_CardSelection
//=============================================================================
// #region

Window_CardSelection.prototype.pushToSelectedCards = function (card) {
	var selectedCards = this._cardsSelected;
	var index = selectedCards.indexOf(card);
	if (index > -1) {
		selectedCards.splice(index, 1);
	}
	else {
		selectedCards.push(card);
	}
	if (selectedCards.length == BattleManager._cardsToSelect) {
		BattleManager._cardsToSelect = 0;
		this._indexesSelected = [];
		BattleManager._cardTypeToSelect = undefined;
		return true;
	}
	else {
		return false;
	}
}

// #endregion


//=============================================================================
// Game_Action
//=============================================================================
// #region

Myth.CGC.Summons.Game_Action_setSubject = Game_Action.prototype.setSubject;
Game_Action.prototype.setSubject = function (subject) {
	if (subject.isCreature())
	{
		this._fieldSubject = subject;
	}
	else
	{
		this.clearFieldSubject();
		Myth.CGC.Summons.Game_Action_setSubject.call(this, subject);
	}
};

Game_Action.prototype.clearFieldSubject = function ()
{
	this._fieldSubject = null;
}

Myth.CGC.Summons.Game_Action_subject = Game_Action.prototype.subject;
Game_Action.prototype.subject = function () {
	if (this._fieldSubject)
		return this._fieldSubject;
	return Myth.CGC.Summons.Game_Action_subject.call(this);
};

Game_Action.prototype.isForNonFieldOnly = function () {
	var item = this.item();
	if (item._summonScope && item._summonScope.includes("creature"))
		return false;
	return true;
};

Game_Action.prototype.cardActionSubject = function () {
	var subject = this.subject();
	if (subject.isCreature())
		return subject._summoner;

	return subject;
};

Myth.CGC.Summons.Game_Actions_randomTarget = Game_Action.prototype.randomTarget;
Game_Action.prototype.randomTarget = function ()
{
	BattleManager._tempSubject = this.subject();
	var target = Myth.CGC.Summons.Game_Actions_randomTarget.call(this);
	BattleManager._tempSubject = null;
	return target;
}

Myth.CGC.Summons.Game_Action_targetsForOpponents = Game_Action.prototype.targetsForOpponents;
Game_Action.prototype.targetsForOpponents = function () {
	var targets = Myth.CGC.Summons.Game_Action_targetsForOpponents.call(this);
	if (this.isForAll()) {
		var unit = this.opponentsUnit();
		if (BattleManager.fieldTargetMode == 'all')
			targets = unit.aliveMembersAndCreatures();
		else if (BattleManager.fieldTargetMode == 'battler')
			targets = unit.aliveMembers();
		else if (BattleManager.fieldTargetMode == 'creature')
			targets = unit.fieldCreatures();

		targets = unit.filterOutUnselectables(this.cardActionSubject(), targets);
	}

	
	return targets;
}

Myth.CGC.Summons.Game_Action_targetsForFriends = Game_Action.prototype.targetsForFriends;
Game_Action.prototype.targetsForFriends = function () {
	var targets = Myth.CGC.Summons.Game_Action_targetsForFriends.call(this);
	if (this.isForAll()) {
		var unit = this.friendsUnit();
		if (BattleManager.fieldTargetMode == 'all')
			targets = unit.aliveMembersAndCreatures();
		else if (BattleManager.fieldTargetMode == 'battler')
			targets = unit.aliveMembers();
		else if (BattleManager.fieldTargetMode == 'creature')
			targets = unit.fieldCreatures();

		targets = unit.filterOutUnselectables(this.cardActionSubject(), targets);
	}

	
	return targets;
}

Myth.CGC.Summons.Game_Action_targetsForEveryone = Game_Action.prototype.targetsForEveryone;
Game_Action.prototype.targetsForEveryone = function ()
{
	var targets = Myth.CGC.Summons.Game_Action_targetsForEveryone.call(this);
	if (this.isForAll())
	{
		targets = this.targetsForFriends().concat(this.targetsForOpponents());
	}

	return targets;
}

// #endregion

//=============================================================================
// DataManager
//=============================================================================
// #region

DataManager.getStatDisplays = function (enemyId)
{
	var dataEnemy = $dataEnemies[enemyId];
	var dataStatDisplays = dataEnemy._creatureStatDisplays;
	var dataStatHides = dataEnemy._creatureStatHides;
	if (dataStatDisplays == undefined) dataStatDisplays = [];
	if (dataStatHides == undefined) dataStatHides = [];
	var creatureStats = Myth.CGC.Summons.creatureStats;
	var creatureStatsToDisplay = [];
	for (var i = 0; i < creatureStats.length; i++)
	{
		if (dataStatHides.includes(creatureStats[i].param))
			continue;

		if (creatureStats[i].alwaysShow)
			creatureStatsToDisplay.push(creatureStats[i]);
		else if (dataStatDisplays.includes(creatureStats[i].param))
		{
			creatureStatsToDisplay.push(creatureStats[i]);
		}
	}
	return creatureStatsToDisplay;
}

Myth.CGC.Summons.DataManager_isDatabaseLoaded = DataManager.isDatabaseLoaded;
DataManager.isDatabaseLoaded = function () {
	if (!Myth.CGC.Summons.DataManager_isDatabaseLoaded.call(this)) return false;
	if (!Myth.loaded_CGC_Summons) {
		this.processFieldSkillNotetags($dataSkills);
		this.processFieldSkillNotetags($dataEnemies);

		this.processFieldCreatureNotetags($dataSkills);

		this.processFieldCreaturePassives($dataEnemies);
		this.processStatDisplayNotetags($dataEnemies);

		this.processFieldScopeNotetags($dataSkills);
		this.processFieldScopeNotetags($dataItems);


		this.processFieldCardPassiveNotetags($dataSkills);

		this.processSummonEnemySpriteNotetags($dataEnemies);

		Myth.loaded_CGC_Summons = true;
	}

	return true;
};

DataManager.processFieldSkillNotetags = function (group) {
	var startNote = /<Field Skills?>/i;
	var endNote = /<\/Field Skills?>/i;

	for (var n = 1; n < group.length; n++) {
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);
		var mode = '';
		for (var i = 0; i < notedata.length; i++) {
			var line = notedata[i];
			if (line.match(startNote)) {
				mode = 'field skills';
				obj._fieldSkills = [];
			}
			else if (line.match(endNote)) {
				mode = '';
			}
			else if (mode == 'field skills') {
				if (line.match(/skill (\d+):? limit (\d+) per (battle|turn)/i)) {
					var fieldSkill = {
						skillId: Number(RegExp.$1),
						limit: Number(RegExp.$2),
						limitType: RegExp.$3,
					}
					obj._fieldSkills.push(fieldSkill);
				}
				else if (line.match(/skill (\d+)/i)) {
					var fieldSkill = {
						skillId: Number(RegExp.$1),
						limit: -1,
						limitType: ""
					}
					obj._fieldSkills.push(fieldSkill);
				}
			}
		}
	}
}

DataManager.processFieldCreatureNotetags = function (group) {
	for (var n = 1; n < group.length; n++) {
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);
		var mode = '';
		for (var i = 0; i < notedata.length; i++) {
			var line = notedata[i];
			if (line.match(/<?field creature id:? (\d+)>?/i))
			{
				obj._fieldCreature = Number(RegExp.$1);
			}
			else if (line.match(/<?summon animation id:? (\d+)>?/i))
			{
				obj._summonAnimation = Number(RegExp.$1);
			}
		}
	}
}

DataManager.processFieldCreaturePassives = function (group) {
	var startNote = /<Creature Passives?>/i;
	var endNote = /<\/Creature Passives?>/i;
	for (var n = 1; n < group.length; n++) {
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);
		var mode = '';
		for (var i = 0; i < notedata.length; i++) {
			var line = notedata[i];
			if (line.match(startNote)) {
				mode = 'passives';
				obj._creaturePassives = {};
			}
			else if (line.match(endNote)) {
				mode = '';
			}
			else if (mode == 'passives') {
				if (line.match(/first turn (?:attack|move):? ?(true|false)/i)) {
					obj._creaturePassives._canAttackFirstTurn = JSON.parse(RegExp.$1);
				}
				if (line.match(/ignore action patterns for player/i)) {
					obj._creaturePassives._ignoreActionPatternsPlayer = true;
				}
			}
		}
	}
};

DataManager.processStatDisplayNotetags = function (group) {
	for (var n = 1; n < group.length; n++) {
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);
		for (var i = 0; i < notedata.length; i++) {
			var line = notedata[i];
			if (line.match(/<Show (?:Stat|param) (\w*)>/i)) {
				if (obj._creatureStatDisplays == undefined)
					obj._creatureStatDisplays = [];
				obj._creatureStatDisplays.push(RegExp.$1);
			}
			if (line.match(/<Hide (?:Stat|param) (\w*)>/i)) {
				if (obj._creatureStatHides == undefined)
					obj._creatureStatHides = [];
				obj._creatureStatHides.push(RegExp.$1);
			}
		}
	}
}

DataManager.processFieldScopeNotetags = function (group) {
	var startNote = /<Card Summons? Scope>/i;
	var endNote = /<\/card summons? scope>/i;
	for (var n = 1; n < group.length; n++) {
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);
		var mode = '';
		for (var i = 0; i < notedata.length; i++) {
			var line = notedata[i];
			if (line.match(startNote)) {
				mode = 'summon scope';
				obj._summonScope = [];
			}
			else if (line.match(endNote)) {
				mode = '';
			}
			else if (mode == 'summon scope')
			{
				if (line.match(/if (.+)/i))
				{
					var conditional = RegExp.$1;
					obj._summonScope.push("conditional " + conditional);
				}
				if (line.match(/(?:Battler|Enemy)/i)) {
					obj._summonScope.push("battler");
				}
				else if (line.match(/(?:Summon|Creature)/i)) {
					obj._summonScope.push("creature");
				}
			}
		}
	}
}

DataManager.processFieldCardPassiveNotetags = function (group) {
	var startNote = /<Card Passives?>/i;
	var endNote = /<\/Card Passives?>/i;

	for (n = 1; n < group.length; n++) {
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);
		var mode = '';
		for (var i = 0; i < notedata.length; i++) {
			var line = notedata[i];
			if (line.match(startNote)) {
				mode = 'passives';
				if (!obj._cardPassives)
					obj._cardPassives = {};
			}
			else if (line.match(endNote)) {
				mode = '';
			}
			else if (mode == 'passives') {
				if (line.match(/start of turn field:? ?(?:skill)? ?(\d+)\s*$/i)) {
					if (!obj.__forcedActions)
						obj.__forcedActions = {};
					if (!obj.__forcedActions.fieldTurnStart)
						obj.__forcedActions.fieldTurnStart = [];
					obj.__forcedActions.fieldTurnStart.push(RegExp.$1);
				}
				else if (line.match(/end of turn field:? ?(?:skill)? ?(\d+)\s*$/i)) {
					if (!obj.__forcedActions)
						obj.__forcedActions = {};
					if (!obj.__forcedActions.fieldTurnEnd)
						obj.__forcedActions.fieldTurnEnd = [];
					obj.__forcedActions.fieldTurnEnd.push(RegExp.$1);
				}
			}
		}
	}
}


Myth.CGC.Summons.Datamanager_processCardPassiveNotetags = DataManager.processCardPassiveNotetags;
DataManager.processCardPassiveNotetags = function (group)
{
	Myth.CGC.Summons.Datamanager_processCardPassiveNotetags.call(this, group);

	var startNote = /<Card Passives?>/i;
	var endNote = /<\/Card Passives?>/i;

	for (n = 1; n < group.length; n++)
	{
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);
		var mode = '';
		for (var i = 0; i < notedata.length; i++)
		{
			var line = notedata[i];
			if (line.match(startNote))
			{
				mode = 'passives';
				if (obj._cardPassives == undefined)
					obj._cardPassives = {};
			}
			else if (line.match(endNote))
			{
				mode = '';
			}
			else if (mode == 'passives')
			{
				if (line.match(/enter (\w*):? ?(?:skill)? ?(\d+) on this/i))
				{
					if (!obj.__forcedActions)
						obj.__forcedActions = {};
					if (!obj.__forcedActions.enterZone)
						obj.__forcedActions.enterZone = [];
					var zoneData = { zone: RegExp.$1, skill: RegExp.$2, target: "this" };
					obj.__forcedActions.enterZone.push(zoneData);
				}
				else if (line.match(/start of turn:? ?(?:skill)? ?(\d+) on this/i))
				{
					if (!obj.__forcedActions)
						obj.__forcedActions = {};
					if (!obj.__forcedActions.turnStart)
						obj.__forcedActions.turnStart = [];
					var data = {skill: Number(RegExp.$1), target: "this"};
					obj.__forcedActions.turnStart.push(data);
				}
				else if (line.match(/end of turn:? ?(?:skill)? ?(\d+) on this/i))
				{
					if (!obj.__forcedActions)
						obj.__forcedActions = {};
					if (!obj.__forcedActions.turnEnd)
						obj.__forcedActions.turnEnd = [];
					var data = {skill: Number(RegExp.$1), target: "this"};
					obj.__forcedActions.turnEnd.push(data);
				}
			}
		}
	}
}

DataManager.processSummonEnemySpriteNotetags = function (group)
{
	for (var n = 1; n < group.length; n++)
	{
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);
		for (var i = 0; i < notedata.length; i++)
		{
			var line = notedata[i];
			if (line.match(/<status window offset x:? ?\+?(-?\d+)>/i))
			{
				var offset = Number(RegExp.$1);
				if (obj._statusWindowOffset == undefined)
					obj._statusWindowOffset = {};
				obj._statusWindowOffset.x = offset;
			}
			else if (line.match(/<status window offset y:? ?\+?(-?\d+)>/i))
			{
				var offset = Number(RegExp.$1);
				if (obj._statusWindowOffset == undefined)
					obj._statusWindowOffset = {};
				obj._statusWindowOffset.y = offset;
			}
			else if (line.match(/<status window offset:? ?\+?(-?\d+) \+?(-?\d+)>/i)) //X and Y at once
			{
				var xoffset = Number(RegExp.$1);
				var yoffset = Number(RegExp.$2);
				if (obj._statusWindowOffset == undefined)
					obj._statusWindowOffset = {};
				obj._statusWindowOffset.x = xoffset;
				obj._statusWindowOffset.y = yoffset;
			}
			else if (line.match(/<grid offset x:? ?\+?(-?\d+)>/i))
			{
				var offset = Number(RegExp.$1);
				if (obj._gridOffset == undefined)
					obj._gridOffset = {};
				obj._gridOffset.x = offset;
			}
			else if (line.match(/<grid offset y:? ?\+?(-?\d+)>/i))
			{
				var offset = Number(RegExp.$1);
				if (obj._gridOffset == undefined)
					obj._gridOffset = {};
				obj._gridOffset.y = offset;
			}
			else if (line.match(/<grid offset:? ?\+?(-?\d+) \+?(-?\d+)>/i)) //X and Y at once
			{
				var xoffset = Number(RegExp.$1);
				var yoffset = Number(RegExp.$2);
				if (obj._gridOffset == undefined)
					obj._gridOffset = {};
				obj._gridOffset.x = xoffset;
				obj._gridOffset.y = yoffset;
			}
			else if (line.match(/<face:? ? (.*) (\d)>/i))
			{
				obj._enemyMugshot = {
					face: RegExp.$1,
					index: Number(RegExp.$2)
				}
			}
		}
	}
}

// #endregion


//This mirrors PartyUI_A but does not* require it.
//====================================================
// Window_EnemyStatus
//====================================================
// #region

function Window_EnemyStatus() {
	this.initialize.apply(this, arguments);
};

Window_EnemyStatus.prototype = Object.create(Window_BattleStatus.prototype);
Window_EnemyStatus.prototype.constructor = Window_EnemyStatus;

Window_EnemyStatus.prototype.initialize = function (enemyWindow)
{
	this._y = 0;
	var coords = Myth.CGC.Summons.coordinates;
	var wx = coords.enemyStatusWindow.x;
	var wy = coords.enemyStatusWindow.y;
	if (Myth.Util.usingMZ) {
		const ww = this.windowWidth();
		const wh = this.windowHeight();
		//const wx = 0;
		//const wy = Graphics.boxHeight - wh;
		const rect = new Rectangle(wx, wy, ww, wh);
		Window_BattleStatus.prototype.initialize.call(this, rect);
	}
	else
		Window_BattleStatus.prototype.initialize.call(this);
	this.open();
	this.x = wx;
	this.y = wy;

	this._enemyWindow = enemyWindow;
	this._enemyWindow._enemyStatusWindow = this;
}

Window_EnemyStatus.prototype.update = function ()
{
	Myth.CGC.Window_BattleStatus_update.call(this);
};

Window_EnemyStatus.prototype.numVisibleRows = function () {
	return 1;
};

Window_EnemyStatus.prototype.actor = function (index)
{
	return $gameTroop.members()[index];
};

if (Myth.Util.usingMZ)
{
	Window_EnemyStatus.prototype.refreshCursor = function ()
	{
		if (this.index() >= 0)
		{
			this.setCursorRect(0, 0, this.contents.width, this.contents.height);
		} else
		{
			this.setCursorRect(0, 0, 0, 0);
		}
	};
}
else
{
	Window_EnemyStatus.prototype.updateCursor = function ()
	{
		if (this.index() >= 0)
		{
			this.setCursorRect(0, 0, this.contents.width, this.contents.height);
		} else
		{
			this.setCursorRect(0, 0, 0, 0);
		}
	};

	Window_EnemyStatus.prototype.drawItem = function (index)
	{
		var actor = $gameTroop.members()[index];
		this.drawBasicArea(this.basicAreaRect(index), actor);
		this.drawGaugeArea(this.gaugeAreaRect(index), actor);
	};

/*	Window_EnemyStatus.prototype.itemRect = function (index)
	{
		return new Rectangle(0, 0, this.contents.width, this.contents.height);
	}*/
}

Window_EnemyStatus.prototype.windowWidth = function () {
	return Myth.CGC.Summons.coordinates.enemyStatusSize.width;
};

Window_EnemyStatus.prototype.maxPageItems = function () {
	return 1;
};

Window_EnemyStatus.prototype.windowHeight = function () {
	var height = Myth.CGC.Summons.coordinates.enemyStatusSize.height;
	return height;
};

Window_EnemyStatus.prototype.topIndex = function () {
	return (Math.max(this.index(), 0));
}

Window_EnemyStatus.prototype.drawAllItems = function () {
	var index = Math.max(this.index(), 0);
	index = Math.min($gameTroop.members().length - 1, index);
	if (index >= 0)
		this.drawItem(index);
};

Window_EnemyStatus.prototype.basicAreaRect = function (index) {
	var rect = this.itemRectForText(0);
	rect.x = 0;
	rect.width = this.contents.width;
	return rect;
};

Window_EnemyStatus.prototype.gaugeAreaRect = function (index) {
	var rect = this.itemRectForText(1);
	rect.width = this.contents.width;
	return rect;
};

Window_EnemyStatus.prototype.drawCurrentAndMax = function (current, max, x, y,
	width, color1, color2) {
	var labelWidth = this.textWidth('HP');
	var valueWidth = this.textWidth('0000');
	var slashWidth = this.textWidth('/');
	var x1 = x + width - valueWidth;
	var x2 = x1 - slashWidth;
	var x3 = x2 - valueWidth;
	if (x3 >= x + labelWidth) {
		this.changeTextColor(color1);
		this.drawText(current, x3, y, valueWidth, 'right');
		this.changeTextColor(color2);
		this.drawText('/', x2, y, slashWidth, 'right');
		this.drawText(max, x1, y, valueWidth, 'right');
	} else {
		this.changeTextColor(color1);
		this.drawText(current, x1, y, valueWidth, 'right');
	}
};

Window_EnemyStatus.prototype.drawGaugeAreaWithTp = function (rect, actor) {
	var halfWidth = this.contents.width * 0.6;
	this.drawActorHp(actor, 0, rect.y + this.lineHeight(), halfWidth);
	this.drawActorMp(actor, 0, rect.y + this.lineHeight() * 2, halfWidth);
	this.drawActorTp(actor, 0, rect.y + this.lineHeight() * 3, halfWidth);
};

Window_EnemyStatus.prototype.drawGaugeAreaWithoutTp = function (rect, actor) {
	var halfWidth = this.contents.width * 0.6;
	this.drawActorHp(actor, 0, rect.y + this.lineHeight(), halfWidth);
	this.drawActorMp(actor, 0, rect.y + this.lineHeight() * 2, halfWidth);
};

Window_EnemyStatus.prototype.drawActorCardZonesVertical = function (actor, x, y) {
	var xx = x;
	var yy = y;
	var textWidth = this.textWidth("000");

	this.drawIcon(Myth.CGC.statusIcons.handSize, xx, yy + 2);
	this.drawText("?", xx + textWidth, yy, textWidth, 'right');

	yy += this.lineHeight();

	this.drawIcon(Myth.CGC.statusIcons.deckSize, xx, yy + 2);
	this.drawText("?", xx + textWidth, yy, textWidth, 'right')


	yy += this.lineHeight();

	this.drawIcon(Myth.CGC.statusIcons.discardSize, xx, yy + 2);
	this.drawText("?", xx + textWidth, yy, textWidth, 'right');
};

Window_EnemyStatus.prototype.drawBasicArea = function (rect, actor)
{
	this.drawItemImage(0);

	var halfWidth = this.contents.width * 0.5;
	this.drawActorName(actor, 0, 0, halfWidth);

	this.drawActorIcons(actor, 0, rect.y + this.lineHeight(), Window_Base._iconWidth * 5);
	this.drawActorCardZonesVertical(actor, this.contents.width - 96, ($dataSystem.optDisplayTp ? this.lineHeight() * 2 : this.lineHeight()));
};

Window_EnemyStatus.prototype.drawItemImage = function (index)
{
	var displayMode = Myth.CGC.Summons.coordinates.enemyDisplayMode;
	if (displayMode == "None") return;

	var enemy = this.actor(index);
	var name = enemy.battlerName();
	var hue = enemy.battlerHue();
	var bitmap;

	
	

	var offset = $dataEnemies[enemy.enemyId()]._statusWindowOffset;
	if (offset == undefined) offset = { x: 0, y: 0 };
	if (offset.x == undefined) offset.x = 0;
	if (offset.y == undefined) offset.y = 0;

	var mugshot = $dataEnemies[enemy.enemyId()]._enemyMugshot;
	if (mugshot)
	{
		const rect = Myth.Util.usingMZ ? this.faceRect(index) : this.basicAreaRect(index);
		this.drawFace(mugshot.face, mugshot.index, offset.x, offset.y, rect.width, rect.height);
		return;
	}
	else
	{
		bitmap = this.loadBitmap(name);
	}
	if (!bitmap.isReady())
	{
		setTimeout(() =>
		{
			this.refresh();
		}, 100);
		return;
	}

	
	if (displayMode == "Full Image")
	{

		this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, offset.x, offset.y);
	}
	else if (displayMode == "Mugshot")
	{
		const rect = Myth.Util.usingMZ ? this.faceRect(index) : this.basicAreaRect(index);
		this.contents.blt(bitmap, offset.x, offset.y, rect.width, rect.height, rect.x, rect.y + this.windowHeight() / 4 - this.lineHeight() * 0.5);
		
	}
/*		const actor = this.actor(index);
		const rect = this.faceRect(index);

		if (Myth.CGC.PartyUI.showActorPortActor) this.drawActorFace(actor, rect.x, rect.y + this.windowHeight() / 4 - this.lineHeight() * 0.5, rect.width, rect.height);
*/
};

Window_EnemyStatus.prototype.loadBitmap = function (name)
{
	var bitmap;
	if ($gameSystem.isSideView())
	{
		bitmap = ImageManager.loadSvEnemy(name);
	} else
	{
		bitmap = ImageManager.loadEnemy(name);
	}
	return bitmap;
}

if (Myth.Util.usingMZ) {
	Window_EnemyStatus.prototype.faceRect = function (index) {
		const rect = this.itemRect(index);
		rect.width = Math.min(this.innerWidth);//, ImageManager.faceWidth);
		rect.height = this.lineHeight() * 1.5;
		rect.x = ImageManager.faceWidth * 0.69;
		return rect;
	};

	Window_EnemyStatus.prototype.basicGaugesX = function (rect) {
		return rect.x - 8;
	};

	Window_EnemyStatus.prototype.placeActorName = function (actor, x, y)
	{
		const key = "enemy%1-name".format(actor.name());
		const sprite = this.createInnerSprite(key, Sprite_Name);
		sprite.setup(actor);
		sprite.move(x, y);
		sprite.show();
	};

	Window_EnemyStatus.prototype.placeStateIcon = function (actor, x, y)
	{
		const key = "enemy%1-stateIcon".format(actor.name());
		const sprite = this.createInnerSprite(key, Sprite_StateIcon);
		sprite.setup(actor);
		sprite.move(x, y);
		sprite.show();
	};

	Window_EnemyStatus.prototype.placeGauge = function (actor, type, x, y)
	{
		const key = "enemy%1-gauge-%2".format(actor.name(), type);
		const sprite = this.createInnerSprite(key, Sprite_Gauge);
		sprite.setup(actor, type);
		sprite.move(x, y);
		sprite.show();
	};

	Window_EnemyStatus.prototype.drawItemStatus = function (index) {
		const actor = this.actor(index);
		const rect = this.itemRectWithPadding(index);
		const nameX = this.nameX(rect);
		const nameY = this.nameY(rect);
		const stateIconX = this.stateIconX(rect);
		const stateIconY = this.stateIconY(rect);
		const basicGaugesX = this.basicGaugesX(rect);
		const basicGaugesY = this.basicGaugesY(rect);
		this.placeTimeGauge(actor, nameX - 8, nameY - this.lineHeight() * 0.5);
		this.placeActorName(actor, nameX - 8, nameY - this.lineHeight() * (($dataSystem.optDisplayTp) ? 1.2 : 1));
		this.placeStateIcon(actor, nameX + 100, nameY - this.lineHeight() * 1.5 + ImageManager.iconHeight * 0.72);
		this.placeBasicGauges(actor, basicGaugesX, basicGaugesY);
		this.drawActorCardZonesVertical(actor, rect.width - this.textWidth("000000"), nameY);
	};

	Myth.CGC.Summons.Sprite_Gauge_bitmapWidth = Sprite_Gauge.prototype.bitmapWidth;
	Sprite_Gauge.prototype.bitmapWidth = function ()
	{
		var isEnemyStatus = false;
		var width;
		if (Myth.CGC.PartyUI)	//We want the original bitmapWidth without the resize command
			width = Myth.CGC.PartyUI.Sprite_Gauge_bitmapWidth.call(this);
		else
			width = Myth.CGC.Summons.Sprite_Gauge_bitmapWidth.call(this);
		if (this.parent)
		{
			var parentWindow = this.parent.parent;

			if (parentWindow)
			{
				if (parentWindow instanceof Window_EnemyStatus)
				{
					width = parentWindow.width * 0.5;
					width = Math.floor(width);
					if (this.bitmap && width != this.bitmap.width)
					{
						this.bitmap.resize(width, this.bitmap.height);
						this.width = width;
						//this.redraw();
					}
					isEnemyStatus = true;
				}
			}
		}

		if (!isEnemyStatus)
			width = Myth.CGC.Summons.Sprite_Gauge_bitmapWidth.call(this);

		return width;
	};

	Window_EnemyStatus.prototype.itemRect = function (index) {
		var rect = Window_BattleStatus.prototype.itemRect.call(this, 0);
		rect.y -= this.lineHeight();
		rect.width = this.innerWidth;
		if ($dataSystem.optDisplayTp) rect.height = this.lineHeight() * 6;
		return rect;
	}
}

Window_EnemyStatus.prototype.isCurrentItemEnabled = function ()
{
	var enemy = this.actor(0);
	var isEnabled = enemy.isEnabledForTargeting();
	return isEnabled;
}

// #region touch input

Window_EnemyStatus.prototype.processTouch = function ()
{
	if (this.canTouchToActivate())
	{
		SceneManager._scene.startEnemyWindowSelection();
		SceneManager._scene._fieldWindow.deactivate();
		SceneManager._scene._fieldWindow.select(-1);
		TouchInput.clear();
	}
	Window_BattleStatus.prototype.processTouch.call(this);
}

Window_EnemyStatus.prototype.canTouchToActivate = function ()
{

	if (!this.isOpen()) return false;
	if (!this.visible) return false;
	if (this.active) return false;
	if (SceneManager._scene._fieldWindow.canMoveToUnitWindow() != "troop") return false;
	if (SceneManager._scene._fieldSkillWindow.isOpenAndActive()) return false;
	if (SceneManager._scene._skillWindow.isOpenAndActive()) return false;
	if (BattleManager._phase != 'input') return false;

	if (Myth.Util.usingMZ)
	{
		if (this.hitIndex() < 0) return false;
		if (!this.isHoverEnabled()) return false;
		if (!TouchInput.isClicked()) return false;
	}
	else
	{
		if (!TouchInput.isTriggered()) return false;
		if (!this.isTouchedInsideFrame()) return false;

	}
	return true;
}

Window_EnemyStatus.prototype.hitTest = function (x, y)
{
	if (this.isTouchedInsideFrame())
	{
		return 0;
	}
	return -1;
}

if (Myth.BLK)
{
	Window_EnemyStatus.prototype.drawActorIcons = function (enemy, x, y, width)
	{
		var iconArray = enemy.allIcons();
		var iconWidth = (Myth.Util.usingMZ ? ImageManager.iconWidth : Window_Base._iconWidth);
		var iconHeight = (Myth.Util.usingMZ ? ImageManager.iconHeight : Window_Base._iconHeight);

		for (var k = 0; k < Myth.BLK._blockStatesList.length; k++) {
			const bsList = Myth.BLK._blockStatesList[k];
			if (bsList) {
				var stateID = bsList.blockStateID;

				if (enemy.isStateAffected(stateID) && $dataEnemies[enemy._enemyId]._hideBlock[stateID])
				{
					iconArray = iconArray.filter((item) => { return item != $dataStates[stateID].iconIndex });
				}
			}
		}

		for (var j = 0; j < Myth.BLK._rescStatesList.length; j++) {
			const rsList = Myth.BLK._rescStatesList[j];
			if (rsList)
			{
				var stateID = rsList.rescStateID;

				if (enemy.isStateAffected(stateID) && $dataEnemies[enemy._enemyId]._hideResc[stateID])
				{
					iconArray = iconArray.filter((item) => { return item != $dataStates[stateID].iconIndex });
				}
			}
		}

		var icons = iconArray.slice(0, Math.floor(width / iconWidth));

		for (var i = 0; i < icons.length; i++) {
			this.drawIcon(icons[i], x + (iconWidth * i), y + 2);

			for (var j = 0; j < Myth.BLK._blockStatesList.length; j++) {
				const bsList = Myth.BLK._blockStatesList[j];
				if (bsList)
				{
					var stateID = bsList.blockStateID;

					if (icons[i] == $dataStates[stateID].iconIndex && enemy.isStateAffected(stateID))
					{
						this.contents.fontSize = bsList.blockIcon.iconFontSize;
						this.contents.fontFace = bsList.blockIcon.iconFontFace;
						this.drawText(String(enemy.getBlock(stateID)),
							x + (iconWidth * i) + ((3 - String(enemy.getBlock(stateID)).length) * 4) + 2,
							y + 2, this.textWidth("000"), iconHeight, 'center');
						this.resetFontSettings();
					}
				}
			}

			for (var k = 0; k < Myth.BLK._rescStatesList.length; k++)
			{
				const rsList = Myth.BLK._rescStatesList[k];
				if (rsList)
				{
					var stateID = rsList.rescStateID;

					if (icons[i] == $dataStates[stateID].iconIndex && enemy.isStateAffected(stateID))
					{
						this.contents.fontSize = rsList.rescIcon.iconFontSize;
						this.contents.fontFace = rsList.rescIcon.iconFontFace;

						var evalValue = eval("enemy." + String(Myth.BLK._rescStatesList[k].rescType).toLowerCase());

						this.drawText(String(evalValue), x + (iconWidth * i) + ((3 - String(evalValue).length) * 4) + 2,
							y + 2, this.textWidth("000"), iconHeight, 'center');
						this.resetFontSettings();
					}
				}
			}
		}
	};

	if (Myth.Util.usingMZ)
	{
		Myth.BLK.Window_EnemyStatus_drawItemStatus = Window_EnemyStatus.prototype.drawItemStatus;
		Window_EnemyStatus.prototype.drawItemStatus = function (index)
		{
			Myth.BLK.Window_EnemyStatus_drawItemStatus.call(this, index);

			const enemy = this.actor(index);
			const rect = this.itemRectWithPadding(index);
			var iconWidth = ImageManager.iconWidth;
			var iconHeight = ImageManager.iconHeight;
			var offset = 1;

			for (var j = 0; j < Myth.BLK._blockStatesList.length; j++)
			{
				const bsList = Myth.BLK._blockStatesList[j];
				if (bsList)
				{
					var stateID = bsList.blockStateID;

					if (enemy.isStateAffected(stateID) && !$dataEnemies[enemy._enemyId]._hideBlock[stateID])
					{
						this.contents.fontSize = bsList.blockIcon.iconFontSize;
						this.contents.fontFace = bsList.blockIcon.iconFontFace;

						this.drawIcon($dataStates[stateID].iconIndex,
							iconWidth * (offset - 1),
							this.lineHeight() * ($dataSystem.optDisplayTp ? 2.5 : 2) - iconHeight - 4);

						this.drawText(String(enemy.getBlock(stateID)),
							(iconWidth * (offset - 1)) + (iconWidth * 0.5) - 4,
							this.lineHeight() * ($dataSystem.optDisplayTp ? 1.5 : 1) - 2,
							this.textWidth("000"), iconHeight, 'center');

						this.resetFontSettings();
						offset++;
					}
				}
			}

			for (var k = 0; k < Myth.BLK._rescStatesList.length; k++)
			{
				const rsList = Myth.BLK._rescStatesList[k];
				if (rsList)
				{
					var stateID = rsList.rescStateID;

					if (enemy.isStateAffected(stateID) && !$dataEnemies[enemy._enemyId]._hideResc[stateID])
					{
						this.contents.fontSize = rsList.rescIcon.iconFontSize;
						this.contents.fontFace = rsList.rescIcon.iconFontFace;

						var evalValue = eval("enemy." + String(Myth.BLK._rescStatesList[k].rescType).toLowerCase());

						this.drawIcon($dataStates[stateID].iconIndex,
							iconWidth * (offset - 1),
							this.lineHeight() * ($dataSystem.optDisplayTp ? 2.5 : 2) - iconHeight - 4);

						this.drawText(String(evalValue),
							(iconWidth * (offset - 1)) + ((3 - String(evalValue).length) * 4),
							this.lineHeight() * ($dataSystem.optDisplayTp ? 1.5 : 1) - 2,
							this.textWidth("000"), iconHeight, 'center');

						this.resetFontSettings();
						offset++;
					}
				}
			}
		}
	}
}

// #endregion
// #endregion
